import React, { useState, useEffect } from 'react';
import { getFavorites } from '../services/api';
import { FaHeart } from 'react-icons/fa';

const FavoritePosts = ({ userId }) => {
  const [favorites, setFavorites] = useState([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchFavorites = async () => {
      try {
        setLoading(true);
        const response = await getFavorites(userId);
        setFavorites(response.data);
      } catch (error) {
        console.error('Failed to fetch favorites:', error);
      } finally {
        setLoading(false);
      }
    };
    
    if (userId) {
      fetchFavorites();
    }
  }, [userId]);

  if (loading) {
    return (
      <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
        {[1, 2, 3, 4, 5, 6].map(item => (
          <div key={item} className="bg-gray-100 rounded-xl aspect-square animate-pulse"></div>
        ))}
      </div>
    );
  }

  if (favorites.length === 0) {
    return (
      <div className="text-center py-16">
        <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-red-100 text-red-500 mb-4">
          <FaHeart className="text-2xl" />
        </div>
        <h3 className="text-lg font-medium text-gray-900">暂无收藏内容</h3>
        <p className="mt-1 text-gray-500">你还没有收藏任何笔记</p>
      </div>
    );
  }

  // 模拟瀑布流布局数据
  const waterfallData = favorites.map(post => ({
    ...post,
    height: Math.floor(Math.random() * 100) + 200 // 随机高度
  }));

  return (
    <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
      {waterfallData.map(post => (
        <div 
          key={post.id} 
          className="bg-white rounded-xl overflow-hidden shadow-sm hover:shadow-md transition-shadow"
        >
          <div 
            className="relative bg-gray-200" 
            style={{ height: `${post.height}px` }}
          >
            {/* 占位图片 */}
            <div className="absolute inset-0 bg-gradient-to-br from-pink-100 to-orange-100"></div>
            
            {/* 类型标签 */}
            <div className="absolute top-2 right-2 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded-full">
              {post.type === 'image' ? '图文' : 
               post.type === 'video' ? '视频' : '文档'}
            </div>
            
            {/* 收藏标记 */}
            <div className="absolute bottom-2 right-2 bg-red-500 rounded-full p-1">
              <FaHeart className="text-white text-xs" />
            </div>
          </div>
          
          <div className="p-3">
            <h3 className="font-medium line-clamp-2">{post.title}</h3>
            <div className="flex items-center mt-2 text-xs text-gray-500">
              <span>❤️ 2.5k</span>
              <span className="mx-2">•</span>
              <span>⭐ 156</span>
            </div>
          </div>
        </div>
      ))}
    </div>
  );
};

export default FavoritePosts;