// src/components/HomeFeed.jsx

import React, { useState, useEffect } from 'react'
import { ThumbsUp } from 'lucide-react'
import { fetchPosts, fetchPost } from '../api/posts'
import '../style/HomeFeed.css'

const categories = [
  '推荐','穿搭','美食','彩妆','影视',
  '职场','情感','家居','游戏','旅行','健身'
]

export default function HomeFeed() {
  const [activeCat, setActiveCat] = useState('推荐')
  const [items, setItems]         = useState([])
  const [loading, setLoading]     = useState(true)
  const [error, setError]         = useState(null)

  useEffect(() => {
    async function loadPosts() {
      try {
        const list = await fetchPosts()  // [{id, title, heat, created_at}, …]
        // 为了拿到 media_urls 和 user_id，这里再拉详情
        const detailed = await Promise.all(
          list.map(async p => {
            const d = await fetchPost(p.id)
            return {
              id:     d.id,
              title:  d.title,
              author: `作者 ${d.user_id}`,
              avatar: `https://i.pravatar.cc/40?img=${d.user_id}`,
              img:    d.media_urls?.[0] || '', // 用第一张媒体作为封面
              likes:  d.heat
            }
          })
        )
        setItems(detailed)
      } catch (e) {
        setError(e.message)
      } finally {
        setLoading(false)
      }
    }
    loadPosts()
  }, [])

  return (
    <div className="home-feed">
      {/* 顶部分类 */}
      <nav className="feed-tabs">
        {categories.map(cat => (
          <button
            key={cat}
            className={cat === activeCat ? 'tab active' : 'tab'}
            onClick={() => setActiveCat(cat)}
          >
            {cat}
          </button>
        ))}
      </nav>

      {/* 状态提示 */}
      {loading ? (
        <div className="loading">加载中…</div>
      ) : error ? (
        <div className="error">加载失败：{error}</div>
      ) : (
        /* 瀑布流卡片区 */
        <div className="feed-grid">
          {items.map(item => (
            <div key={item.id} className="feed-card">
              <img className="card-img" src={item.img} alt={item.title} />
              <h3 className="card-title">{item.title}</h3>
              <div className="card-footer">
                <div className="card-author">
                  <img className="avatar" src={item.avatar} alt={item.author} />
                  <span className="username">{item.author}</span>
                </div>
                <div className="card-likes">
                  <ThumbsUp size={16} />
                  <span className="likes-count">{item.likes}</span>
                </div>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  )
}
