// src/features/auth/services/authApi.js
import request from "../../../services/request";
import { message } from "antd";

// 使用API前缀
const API_PREFIX = "/user";
const ADMIN_PREFIX = "/admin";

// 导出API函数
export const loginUser = (credentials) => {
  return request.post(`${API_PREFIX}/login`, credentials).then((response) => {
    if (response.data && response.data.success) {
      // 保存token和用户信息到localStorage
      localStorage.setItem("token", response.data.data.token);
      localStorage.setItem("user", JSON.stringify(response.data.data.user));
      return response.data;
    } else {
      return Promise.reject(new Error(response.data.message || "登录失败"));
    }
  });
};

export const adminLogin = (credentials) => {
  return request.post(`${ADMIN_PREFIX}/login`, credentials).then((response) => {
    if (response.data && response.data.success) {
      // 保存token和用户信息到localStorage
      localStorage.setItem("token", response.data.data.token);
      localStorage.setItem("user", JSON.stringify(response.data.data.user));
      return response.data;
    } else {
      return Promise.reject(new Error(response.data.message || "管理员登录失败"));
    }
  });
};

export const registerUser = (userData) => {
  return request.post(`${API_PREFIX}/register`, userData).then((response) => {
    if (response.data && response.data.success) {
      return response.data;
    } else {
      return Promise.reject(new Error(response.data.message || "注册失败"));
    }
  });
};

export const updateUsername = (username, newUsername) => {
  const token = localStorage.getItem("token");
  return request
    .post(`${API_PREFIX}/update/username`, 
    { username, newUsername }, 
    { headers: { token } })
    .then((response) => {
      if (response.data && response.data.success) {
        // 更新本地存储的用户信息
        const user = JSON.parse(localStorage.getItem("user") || "{}");
        user.username = newUsername;
        localStorage.setItem("user", JSON.stringify(user));
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "修改用户名失败")
        );
      }
    });
};

export const updatePassword = (username, newPassword) => {
  const token = localStorage.getItem("token");
  return request
    .post(`${API_PREFIX}/update/password`, 
    { username, newPassword }, 
    { headers: { token } })
    .then((response) => {
      if (response.data && response.data.success) {
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "修改密码失败")
        );
      }
    });
};

export const updateEmail = (username, newEmail) => {
  const token = localStorage.getItem("token");
  return request
    .post(`${API_PREFIX}/update/email`, 
    { username, newEmail }, 
    { headers: { token } })
    .then((response) => {
      if (response.data && response.data.success) {
        // 更新本地存储的用户信息
        const user = JSON.parse(localStorage.getItem("user") || "{}");
        user.email = newEmail;
        localStorage.setItem("user", JSON.stringify(user));
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "修改邮箱失败")
        );
      }
    });
};

export const getUserInfo = (username) => {
  const token = localStorage.getItem("token");
  return request
    .get(`${API_PREFIX}/get/info?username=${username}`, 
    { headers: { token } })
    .then((response) => {
      if (response.data && response.data.success) {
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "获取用户信息失败")
        );
      }
    });
};

export const getUserList = (username) => {
  const token = localStorage.getItem("token");
  return request
    .get(`/user/list?username=${username}`, 
    { headers: { token } })
    .then((response) => {
      if (response.data && response.data.success) {
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "获取用户列表失败")
        );
      }
    });
};

export const deleteUser = (username, targetUsername) => {
  const token = localStorage.getItem("token");
  return request
    .delete(`/user/delete`, 
    { 
      headers: { token },
      data: { username, targetUsername }
    })
    .then((response) => {
      if (response.data && response.data.success) {
        return response.data;
      } else {
        return Promise.reject(
          new Error(response.data.message || "删除用户失败")
        );
      }
    });
};

export const logoutUser = () => {
  // 清除本地存储
  localStorage.removeItem("token");
  localStorage.removeItem("user");

  return Promise.resolve({
    success: true,
    message: "注销成功"
  });
};
