// src/features/auth/pages/LoginPage.jsx
import React, { useState } from "react";
import { useNavigate, Link } from "react-router-dom";
import {
  Form,
  Input,
  Button,
  Checkbox,
  Card,
  Typography,
  Space,
  Divider,
  message,
} from "antd";
import { UserOutlined, LockOutlined } from "@ant-design/icons";
import { useAuth } from "../contexts/AuthContext"; // 使用新的 AuthContext
// import { loginUser } from '../services/authApi'; // 如果不直接在 context 中调用 API

const { Title, Text } = Typography;

const LoginPage = () => {
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();
  const { login, isAuthenticated, user } = useAuth(); // 从 Context 获取 login 方法等

  React.useEffect(() => {
    // 如果已经登录，并且有用户信息，则重定向到首页
    if (isAuthenticated && user) {
      navigate("/");
    }
  }, [isAuthenticated, user, navigate]);

  const onFinish = async (values) => {
    setLoading(true);
    try {
      await login({ username: values.username, password: values.password });
      // 登录成功后的导航由 AuthContext 内部或 ProtectedRoute 处理
      // AuthContext 已经包含成功提示，这里不再重复提示
      navigate("/"); // 或者根据用户角色导航到不同页面
    } catch (error) {
      // 错误消息由 AuthContext 中的 login 方法或 request 拦截器处理
      console.error("Login page error:", error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="flex justify-center items-center min-h-screen bg-slate-100 p-4">
      {" "}
      {/* Tailwind: bg-gray-100 -> bg-slate-100 */}
      <Card className="w-full max-w-md shadow-lg rounded-lg">
        {" "}
        {/* Tailwind: rounded-lg */}
        <div className="text-center mb-8">
          {" "}
          {/* Tailwind: mb-6 -> mb-8 */}
          <Title level={2} className="!mb-2 text-slate-700">
            PT站登录
          </Title>{" "}
          {/* Tailwind: text-slate-700 */}
          <Text type="secondary">欢迎回来，请登录您的账号</Text>
        </div>
        <Form
          name="login_form" // 最好给表单一个唯一的名字
          initialValues={{ remember: true }}
          onFinish={onFinish}
          size="large"
          layout="vertical"
          className="space-y-6" // Tailwind: 间距控制
        >
          <Form.Item
            name="username"
            rules={[{ required: true, message: "请输入您的用户名!" }]}
          >
            <Input
              prefix={<UserOutlined className="site-form-item-icon" />}
              placeholder="用户名"
            />
          </Form.Item>
          <Form.Item
            name="password"
            rules={[{ required: true, message: "请输入您的密码!" }]}
          >
            <Input.Password
              prefix={<LockOutlined className="site-form-item-icon" />}
              placeholder="密码"
            />
          </Form.Item>
          <Form.Item className="!mb-0">
            {" "}
            {/* Tailwind: !mb-0 覆盖antd默认margin */}
            <div className="flex justify-between items-center">
              <Form.Item name="remember" valuePropName="checked" noStyle>
                <Checkbox>记住我</Checkbox>
              </Form.Item>
              <Link
                to="/forgot-password"
                className="text-blue-600 hover:text-blue-700 hover:underline"
              >
                {" "}
                {/* Tailwind: hover:underline */}
                忘记密码?
              </Link>
            </div>
          </Form.Item>
          <Form.Item>
            <Button
              type="primary"
              htmlType="submit"
              className="w-full !text-base"
              loading={loading}
            >
              {" "}
              {/* Tailwind: !text-base (示例) */}登 录
            </Button>
          </Form.Item>
          <Divider plain>
            <span className="text-slate-500">或</span>
          </Divider>{" "}
          {/* Tailwind: text-slate-500 */}
          <div className="text-center">
            <Text type="secondary" className="mr-1">
              还没有账号？
            </Text>
            <Link
              to="/register"
              className="font-medium text-blue-600 hover:text-blue-700 hover:underline"
            >
              立即注册
            </Link>
          </div>
        </Form>
        {/* 提示信息部分可以保留或移除 */}
        <div className="mt-8 p-4 bg-slate-50 rounded-md border border-slate-200">
          {" "}
          {/* Tailwind: border, border-slate-200 */}
          <Text
            type="secondary"
            className="block mb-2 font-semibold text-slate-600"
          >
            测试账号提示
          </Text>
          <ul className="space-y-1 text-sm text-slate-500 list-disc list-inside">
            <li>管理员: admin / admin123</li>
            <li>普通用户: user / user123</li>
            {/* ...其他测试账号 */}
          </ul>
        </div>
      </Card>
    </div>
  );
};

export default LoginPage;
