import React, { useState, useEffect } from 'react';
import { List, Avatar, Space, Tag, Typography, Button, message, Modal, Form, Input, Spin } from 'antd';
import { getPosts, createPost } from '../services/forumApi';

const { Title, Paragraph, Text } = Typography;
const { TextArea } = Input;

const ForumPage = () => {
  const [posts, setPosts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [form] = Form.useForm();
  
  // 获取用户信息
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  
  // 加载帖子数据
  useEffect(() => {
    fetchPosts();
  }, []);
  
  // 获取帖子列表
  const fetchPosts = async () => {
    try {
      setLoading(true);
      const response = await getPosts({ username: user.username });
      if (response.success) {
        setPosts(response.data.posts || []);
      }
    } catch (error) {
      message.error(error.message || '获取帖子列表失败');
    } finally {
      setLoading(false);
    }
  };
  
  // 显示新建帖子对话框
  const showModal = () => {
    setIsModalOpen(true);
  };
  
  // 关闭对话框
  const handleCancel = () => {
    setIsModalOpen(false);
    form.resetFields();
  };
  
  // 提交新帖子
  const handleSubmit = async () => {
    try {
      const values = await form.validateFields();
      
      // 添加作者信息
      values.author = user.username;
      
      const response = await createPost(values);
      if (response.success) {
        message.success('帖子发布成功');
        setIsModalOpen(false);
        form.resetFields();
        fetchPosts(); // 重新加载帖子列表
      }
    } catch (error) {
      message.error(error.message || '发布帖子失败');
    }
  };

  return (
    <div className="space-y-6">
      <Title level={2}>社区论坛</Title>
      <Paragraph className="text-slate-500">
        欢迎来到我们的社区论坛，这里是会员交流分享的地方。
      </Paragraph>
      
      {loading ? (
        <div className="flex justify-center py-8">
          <Spin size="large" tip="加载中..." />
        </div>
      ) : (
        <List
          itemLayout="vertical"
          size="large"
          dataSource={posts}
          renderItem={(item) => (
            <List.Item
              key={item.id}
              extra={
                <Space>
                  <Tag color="green">浏览: {item.views || 0}</Tag>
                  <Tag color="blue">点赞: {item.likes || 0}</Tag>
                  <Text type="secondary">{item.createTime}</Text>
                </Space>
              }
            >
              <List.Item.Meta
                avatar={<Avatar src={`https://api.dicebear.com/7.x/avataaars/svg?seed=${item.author}`} />}
                title={<a href={`/post/${item.id}`}>{item.title}</a>}
                description={<Text type="secondary">作者: {item.author}</Text>}
              />
              <Paragraph ellipsis={{ rows: 2 }}>{item.content}</Paragraph>
            </List.Item>
          )}
        />
      )}
      
      <div className="text-center mt-4">
        <Button type="primary" onClick={showModal}>发布新主题</Button>
      </div>
      
      {/* 新建帖子对话框 */}
      <Modal
        title="发布新主题"
        open={isModalOpen}
        onOk={handleSubmit}
        onCancel={handleCancel}
        okText="发布"
        cancelText="取消"
      >
        <Form
          form={form}
          layout="vertical"
        >
          <Form.Item
            name="title"
            label="标题"
            rules={[{ required: true, message: '请输入标题' }]}
          >
            <Input placeholder="请输入标题" />
          </Form.Item>
          <Form.Item
            name="content"
            label="内容"
            rules={[{ required: true, message: '请输入帖子内容' }]}
          >
            <TextArea rows={6} placeholder="请输入帖子内容" />
          </Form.Item>
        </Form>
      </Modal>
    </div>
  );
};

export default ForumPage; 