import axios from "axios";
import { message } from "antd";

// 创建 axios 实例
const request = axios.create({
  baseURL: "/api",
  timeout: 10000,
});

// 请求拦截器
// 请求拦截器中添加角色检查
request.interceptors.request.use(
  (config) => {
    // 从 localStorage 获取 token
    const token = localStorage.getItem("token");

    // 如果有 token 则添加到请求头
    if (token) {
      config.headers["Authorization"] = `Bearer ${token}`;
    }

    // 角色检查
    // 例如，对特定API路径进行角色检查
    if (config.url.startsWith("/api/admin") && !hasAdminRole()) {
      // 取消请求
      return Promise.reject(new Error("无权限执行此操作"));
    }

    return config;
  },
  (error) => {
    return Promise.reject(error);
  }
);

// 辅助函数：检查是否有管理员角色
function hasAdminRole() {
  const user = JSON.parse(localStorage.getItem("user") || "{}");
  return user.role === "admin";
}

// 响应拦截器
request.interceptors.response.use(
  (response) => {
    return response;
  },
  (error) => {
    if (error.response) {
      const { status, data } = error.response;

      // 处理 401 未授权错误（token 无效或过期）
      if (status === 401) {
        message.error("登录已过期，请重新登录");

        // 清除本地存储的 token 和用户信息
        localStorage.removeItem("token");
        localStorage.removeItem("user");

        // 重定向到登录页
        if (window.location.pathname !== "/login") {
          window.location.href = "/login";
        }
      } else {
        // 处理其他错误
        message.error(data.message || "请求失败");
      }
    } else if (error.request) {
      // 请求发出但没有收到响应
      message.error("网络错误，请检查您的网络连接");
    } else {
      // 请求配置出错
      message.error("请求配置错误");
    }

    return Promise.reject(error);
  }
);

export default request;
