import React, { useState, useEffect } from "react";
import {
  Typography,
  Button,
  Tabs,
  Radio,
  Tag,
  Divider,
  Checkbox,
  Input,
  Table,
  Space,
  Tooltip,
  Badge,
  Spin,
  Skeleton,
  Result,
} from "antd";
import {
  UploadOutlined,
  SearchOutlined,
  DownloadOutlined,
  EyeOutlined,
  LinkOutlined,
  FileTextOutlined,
  CheckCircleOutlined,
  ClockCircleOutlined,
  LoadingOutlined,
  ReloadOutlined,
} from "@ant-design/icons";
import { Link } from "react-router-dom";
import { getTorrentList } from "../../../api/torrents";
import { useAuth } from "../../auth/contexts/AuthContext";

const { Title, Text } = Typography;
const { TabPane } = Tabs;
const { Search } = Input;

// 模拟种子数据
const torrentData = Array.from({ length: 100 }, (_, index) => ({
  key: index + 1,
  id: index + 1,
  category: [
    "电影",
    "剧集",
    "音乐",
    "动漫",
    "游戏",
    "综艺",
    "体育",
    "软件",
    "学习",
    "纪录片",
    "其他",
  ][index % 11],
  subcategory:
    index % 11 === 0
      ? "动作"
      : index % 11 === 1
      ? "美剧"
      : index % 11 === 2
      ? "古典"
      : index % 11 === 3
      ? "日漫"
      : index % 11 === 4
      ? "RPG"
      : index % 11 === 5
      ? "脱口秀"
      : index % 11 === 6
      ? "足球"
      : index % 11 === 7
      ? "工具"
      : index % 11 === 8
      ? "课程"
      : index % 11 === 9
      ? "自然"
      : "其他",
  title: `种子标题 ${index + 1} ${index % 5 === 0 ? "[中字]" : ""} ${
    index % 7 === 0 ? "合集" : ""
  }`,
  size: `${(Math.random() * 50).toFixed(1)} GB`,
  fileCount: Math.floor(Math.random() * 100) + 1,
  views: Math.floor(Math.random() * 1000) + 100,
  publishTime: `2023-${Math.floor(Math.random() * 12) + 1}-${
    Math.floor(Math.random() * 28) + 1
  }`,
  seeders: Math.floor(Math.random() * 100),
  leechers: Math.floor(Math.random() * 50),
  completed: Math.floor(Math.random() * 200),
  uploader: `user${Math.floor(Math.random() * 10) + 1}`,
  isOwnTorrent: index % 15 === 0,
  hasSubtitle: index % 5 === 0,
  isCollection: index % 7 === 0,
  isActive: Math.random() > 0.2, // 80%的概率是活种
}));

// 定义类别及其子类别
const categories = [
  {
    key: "all",
    name: "全部",
    count: torrentData.length,
    subcategories: [],
  },
  {
    key: "movie",
    name: "电影",
    count: torrentData.filter((t) => t.category === "电影").length,
    subcategories: [
      "动作",
      "喜剧",
      "爱情",
      "科幻",
      "恐怖",
      "剧情",
      "战争",
      "纪录",
      "动画",
      "其他",
    ],
  },
  {
    key: "tv",
    name: "剧集",
    count: torrentData.filter((t) => t.category === "剧集").length,
    subcategories: ["美剧", "英剧", "韩剧", "日剧", "国产剧", "港台剧", "其他"],
  },
  {
    key: "music",
    name: "音乐",
    count: torrentData.filter((t) => t.category === "音乐").length,
    subcategories: [
      "流行",
      "摇滚",
      "电子",
      "民谣",
      "嘻哈",
      "古典",
      "爵士",
      "原声带",
      "其他",
    ],
  },
  {
    key: "anime",
    name: "动漫",
    count: torrentData.filter((t) => t.category === "动漫").length,
    subcategories: ["日漫", "国漫", "美漫", "剧场版", "OVA", "其他"],
  },
  {
    key: "game",
    name: "游戏",
    count: torrentData.filter((t) => t.category === "游戏").length,
    subcategories: [
      "角色扮演",
      "动作",
      "射击",
      "策略",
      "模拟",
      "冒险",
      "体育",
      "格斗",
      "其他",
    ],
  },
  {
    key: "variety",
    name: "综艺",
    count: torrentData.filter((t) => t.category === "综艺").length,
    subcategories: [
      "真人秀",
      "脱口秀",
      "访谈",
      "选秀",
      "纪实",
      "搞笑",
      "情感",
      "其他",
    ],
  },
  {
    key: "sports",
    name: "体育",
    count: torrentData.filter((t) => t.category === "体育").length,
    subcategories: [
      "足球",
      "篮球",
      "网球",
      "赛车",
      "拳击",
      "格斗",
      "奥运",
      "其他",
    ],
  },
  {
    key: "software",
    name: "软件",
    count: torrentData.filter((t) => t.category === "软件").length,
    subcategories: [
      "操作系统",
      "应用软件",
      "图形设计",
      "音频编辑",
      "视频制作",
      "编程开发",
      "其他",
    ],
  },
  {
    key: "learning",
    name: "学习",
    count: torrentData.filter((t) => t.category === "学习").length,
    subcategories: [
      "语言",
      "编程",
      "设计",
      "经济",
      "管理",
      "考试",
      "技能",
      "其他",
    ],
  },
  {
    key: "documentary",
    name: "纪录片",
    count: torrentData.filter((t) => t.category === "纪录片").length,
    subcategories: [
      "自然",
      "历史",
      "科学",
      "人文",
      "社会",
      "军事",
      "传记",
      "其他",
    ],
  },
  {
    key: "other",
    name: "其他",
    count: torrentData.filter((t) => t.category === "其他").length,
    subcategories: ["其他"],
  },
];

const TorrentListPage = () => {
  // 状态管理
  const [ownTorrentsOnly, setOwnTorrentsOnly] = useState(false);
  const [searchText, setSearchText] = useState("");
  const [currentPage, setCurrentPage] = useState(1);
  const [pageSize] = useState(50);
  
  // 添加加载状态
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [torrents, setTorrents] = useState([]);
  
  // 获取用户信息
  const { user } = useAuth();

  // 加载种子数据
  useEffect(() => {
    const loadData = async () => {
      if (!user?.username && !user?.uid) {
        setError("用户信息不完整，请重新登录");
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        const username = user?.username || user?.uid;
        const response = await getTorrentList({ username });
        
        if (response && Array.isArray(response)) {
          // 转换数据格式，添加缺失字段的默认值
          const transformedData = response.map((item, index) => ({
            key: item.resourceId || index,
            id: item.resourceId,
            resourceId: item.resourceId,
            title: item.name || "未知标题",
            name: item.name,
            size: item.size || "0 MB",
            publishTime: item.publishTime || "",
            uploader: item.author || "未知作者",
            author: item.author,
            description: item.description || "",
            // 设置默认值
            category: "其他",
            subcategory: "其他",
            fileCount: 1,
            views: 0,
            seeders: 0,
            leechers: 0,
            completed: 0,
            isOwnTorrent: (item.author === username),
            hasSubtitle: false,
            isCollection: false,
            isActive: true,
          }));
          setTorrents(transformedData);
        } else {
          setTorrents([]);
        }
        setError(null);
      } catch (err) {
        console.error("加载种子数据失败:", err);
        setError("加载种子数据失败，请稍后再试");
        setTorrents([]);
      } finally {
        setLoading(false);
      }
    };

    loadData();
  }, [user]);



  // 过滤种子数据
  const filteredTorrents = torrents.filter((torrent) => {
    // 我的种子筛选
    if (ownTorrentsOnly && !torrent.isOwnTorrent) return false;

    // 搜索文本筛选
    if (searchText) {
      const searchLower = searchText.toLowerCase();
      return (
        torrent.title.toLowerCase().includes(searchLower) ||
        torrent.description.toLowerCase().includes(searchLower) ||
        torrent.uploader.toLowerCase().includes(searchLower)
      );
    }

    return true;
  });

  // 处理搜索，并添加加载状态
  const handleSearch = (value) => {
    setLoading(true);
    setTimeout(() => {
      setSearchText(value);
      setLoading(false);
    }, 500);
  };

  // 表格列定义
  const columns = [
    {
      title: "资源名称",
      dataIndex: "title",
      key: "title",
      render: (text, record) => (
        <Space direction="vertical" size={2}>
          <Link to={`/torrent/${record.id}`} style={{ fontWeight: 'bold' }}>
            {text}
          </Link>
          {record.description && (
            <Text type="secondary" style={{ fontSize: "12px" }}>
              {record.description.length > 50 
                ? `${record.description.substring(0, 50)}...` 
                : record.description}
            </Text>
          )}
        </Space>
      ),
    },
    {
      title: "操作",
      key: "action",
      width: 120,
      render: (_, record) => (
        <Space size="small">
          <Tooltip title="下载种子">
            <Button type="primary" size="small" icon={<DownloadOutlined />} />
          </Tooltip>
          <Tooltip title="查看详情">
            <Button size="small" icon={<EyeOutlined />} />
          </Tooltip>
          <Tooltip title="复制链接">
            <Button size="small" icon={<LinkOutlined />} />
          </Tooltip>
        </Space>
      ),
    },
    {
      title: "大小",
      dataIndex: "size",
      key: "size",
      width: 100,
      sorter: (a, b) => {
        // 简单的大小排序，假设格式为 "数字 单位"
        const parseSize = (sizeStr) => {
          const match = sizeStr.match(/^([\d.]+)\s*(\w+)/);
          if (!match) return 0;
          const [, num, unit] = match;
          const multipliers = { 'KB': 1, 'MB': 1024, 'GB': 1024 * 1024, 'TB': 1024 * 1024 * 1024 };
          return parseFloat(num) * (multipliers[unit.toUpperCase()] || 1);
        };
        return parseSize(a.size) - parseSize(b.size);
      },
    },
    {
      title: "发布时间",
      dataIndex: "publishTime",
      key: "publishTime",
      width: 120,
      sorter: (a, b) => new Date(a.publishTime) - new Date(b.publishTime),
      render: (time) => {
        if (!time) return '-';
        try {
          return new Date(time).toLocaleDateString('zh-CN');
        } catch {
          return time;
        }
      },
    },
    {
      title: "发布者",
      dataIndex: "uploader",
      key: "uploader",
      width: 120,
      render: (text, record) => (
        <Space>
          <Link to={`/user/${text}`}>{text}</Link>
          {record.isOwnTorrent && (
            <Tooltip title="我的种子">
              <Tag color="green">我</Tag>
            </Tooltip>
          )}
        </Space>
      ),
    },
  ];
  
  // 加载错误时显示错误信息
  if (error) {
    return (
      <Result
        status="error"
        title="加载失败"
        subTitle={error}
        extra={[
          <Button 
            type="primary" 
            key="reload" 
            icon={<ReloadOutlined />}
            onClick={() => window.location.reload()}
          >
            重新加载
          </Button>
        ]}
      />
    );
  }

  // 自定义加载图标
  const antIcon = <LoadingOutlined style={{ fontSize: 24 }} spin />;

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <Title level={2}>种子列表</Title>
        <Button type="primary" icon={<UploadOutlined />}>
          <Link to="/upload">发布种子</Link>
        </Button>
      </div>

      {/* 筛选条件 */}
      <div className="flex justify-between items-center flex-wrap gap-4">
        <div className="flex items-center gap-4 flex-wrap">
          <Skeleton loading={loading} active paragraph={false} title={{ width: '100%' }} className="inline-block">
            <Checkbox
              checked={ownTorrentsOnly}
              onChange={(e) => setOwnTorrentsOnly(e.target.checked)}
            >
              我的种子
            </Checkbox>
          </Skeleton>
        </div>

        <div>
          <Search
            placeholder="搜索种子名称、描述或发布者"
            allowClear
            enterButton={<SearchOutlined />}
            size="middle"
            onSearch={handleSearch}
            style={{ width: 350 }}
            loading={loading}
          />
        </div>
      </div>

      {/* 种子列表 */}
      <div className="relative">
        <Table
          columns={columns}
          dataSource={filteredTorrents}
          pagination={{
            current: currentPage,
            pageSize: pageSize,
            total: filteredTorrents.length,
            onChange: (page) => {
              setLoading(true);
              setTimeout(() => {
                setCurrentPage(page);
                setLoading(false);
              }, 500);
            },
            showSizeChanger: false,
            showTotal: (total) => `共 ${total} 个种子`,
          }}
          size="middle"
          bordered
          scroll={{ x: 800 }}
          loading={{
            spinning: loading,
            indicator: <Spin indicator={antIcon} />,
            tip: "正在加载种子数据...",
          }}
          locale={{
            emptyText: "没有找到符合条件的种子",
          }}
        />
      </div>
    </div>
  );
};

export default TorrentListPage;
