package com.pt.controller;

import com.pt.constant.Constants;
import com.pt.entity.Post;
import com.pt.service.PostService;
import com.pt.utils.JWTUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/posts")
@CrossOrigin(origins = "*")
public class PostController {

    @Autowired
    private PostService postService;

    /*
        * 创建一个新帖子
        * @param token 用户的JWT令牌
        * @param title 帖子标题
        * @param content 帖子内容
        * @param author 帖子作者（用户名）
        * @return 创建成功的响应
     */
    @PostMapping("/create")
    public ResponseEntity<?> createPost(
            @RequestHeader("token") String token,
            @RequestBody Map<String, String> request
    ) {
        String title = request.get("title");
        String content = request.get("content");
        String author = request.get("author");

        Map<String, Object> ans = new HashMap<>();

        if(!JWTUtils.checkToken(token, author, Constants.UserRole.USER)){
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        Post existingPost = postService.findPostByTitle(title);
        if (existingPost != null) {
            ans.put("result", "Post with this title already exists");
            return ResponseEntity.badRequest().body(ans);
        }

        postService.createPost(title, content, author);
        ans.put("message", "Post created successfully");
        return ResponseEntity.ok(ans);
    }

    /*
        * 获取帖子列表
        * @param token 用户的JWT令牌
        * @param username 用户名
        * @param title 帖子标题（可选）
        * @param author 帖子作者（可选）
        * @param date 帖子发布日期（可选）
     */
    @GetMapping("/list")
    public ResponseEntity<?> getPost(
            @RequestHeader("token") String token,
            @RequestParam("username") String username,
            @RequestParam(value = "title", required = false) String title,
            @RequestParam(value = "author", required = false) String author,
            @RequestParam(value = "date", required = false) String date
    ) {
        Map<String, Object> ans = new HashMap<>();

        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)){
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        List<Post> posts = postService.listAll();

        if(title != null){
            posts.removeIf(post -> !post.getTitle().startsWith(title));
        }

        if(author != null){
            posts.removeIf(post -> !post.getAuthor().startsWith(author));
        }

        if(date != null){
            posts.removeIf(post -> !post.getPublishDate().toString().equals(date));
        }
        ans.put("message", "Post retrieved successfully");
        ans.put("data", Map.of(
                "post", posts
        ));
        return ResponseEntity.ok(ans);
    }

    /*
        * 删除帖子
        * @param token 用户的JWT令牌
        * @param username 用户名
        * @param pid 帖子ID
        * @return 删除成功的响应
     */
    @DeleteMapping("/delete")
    public ResponseEntity<?> deletePost(
            @RequestHeader("token") String token,
            @RequestBody Map<String, String> request
    ) {
        String username = request.get("username");
        int pid = Integer.parseInt(request.get("pid"));

        Map<String, Object> ans = new HashMap<>();

        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)){
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        Post post = postService.findPostById(pid);
        if (post == null) {
            ans.put("message", "Post not found");
            return ResponseEntity.badRequest().body(ans);
        }

        postService.deletePost(post);
        ans.put("message", "Post deleted successfully");
        return ResponseEntity.ok(ans);
    }
}
