package com.pt.controller;

import com.pt.constant.Constants;
import com.pt.entity.User;
import com.pt.service.InvitedCodeService;
import com.pt.utils.JWTUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import com.pt.service.UserService;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/api/user")
@CrossOrigin(origins = "*")
public class UserController {

    @Autowired
    private UserService userService;
    private InvitedCodeService invitedCodeService;

    @PostMapping("/register")
    public ResponseEntity<?> registerUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");
        String email = request.get("email");

        if (username == null || password == null || email == null) {
            return ResponseEntity.badRequest().body("Missing required fields");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            return ResponseEntity.badRequest().body("User already exists");
        } else {
            User newUser = new User();

            if(request.get("invitedCode") != null) {
                String invitedCode = request.get("invitedCode");
                if (!invitedCodeService.isCodeValid(invitedCode)) {
                    return ResponseEntity.badRequest().body("Invalid invited code");
                }
                else {
                    newUser.setLevel(Constants.UserLevel.FRESH.getValue());
                    newUser.setPoints(300);
                }
            }

            String uid = String.valueOf(System.currentTimeMillis());
            newUser.setUid(uid);
            newUser.setUsername(username);
            newUser.setPassword(password);
            newUser.setEmail(email);
            userService.save(newUser);

            Map<String, Object> ans = new HashMap<>();
            ans.put("message", "User registered successfully");
            ans.put("data", newUser);

            return ResponseEntity.ok().body(ans);
        }
    }

    @PostMapping("/login")
    public ResponseEntity<?> loginUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");

        if (username == null || password == null) {
            return ResponseEntity.badRequest().body("Missing username or password");
        }

        User user = userService.findByUsernameAndPassword(username, password);
        Map<String, Object> ans = new HashMap<>();
        if (user != null) {
            String token = JWTUtils.generateToken(username, Constants.UserRole.USER, Constants.DEFAULT_EXPIRE_TIME);
            ans.put("message", "Login successful");
            ans.put("data", token);
            return ResponseEntity.ok().body(ans);
        } else {
            ans.put("message", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/username")
    public ResponseEntity<?> updateUsername(@RequestHeader("token") String token,
                                            @RequestBody Map<String, String> request) {
        String oldUsername = request.get("username");
        String newUsername = request.get("newUsername");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, oldUsername, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(oldUsername);
        if (user != null) {
            user.setUsername(newUsername);
            userService.save(user);
            ans.put("message", "Username updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/password")
    public ResponseEntity<?> updatePassword(@RequestHeader("token") String token,
                                            @RequestBody Map<String, String> request
                                            ) {
        String username = request.get("username");
        String newPassword = request.get("newPassword");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setPassword(newPassword);
            userService.save(user);
            ans.put("message", "Password updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/email")
    public ResponseEntity<?> updateEmail(@RequestHeader("token") String token,
                                         @RequestBody Map<String, String> request) {
        String username = request.get("username");
        String newEmail = request.get("newEmail");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setEmail(newEmail);
            userService.save(user);
            ans.put("message", "Email updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @DeleteMapping("/delete")
    public ResponseEntity<?> deleteUser(@RequestHeader("token") String token,
                                        @RequestBody Map<String, String> request) {
        String username = request.get("username");
        String targetUsername = request.get("targetUsername");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(targetUsername);
        if (user != null) {
            userService.deleteById(user.getUid());
            ans.put("message", "User deleted successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @GetMapping("/list")
    public ResponseEntity<?> listUsers(@RequestHeader("token") String token,
                                       @RequestBody Map<String, String> request) {
        String username = request.get("username");
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        Map<String, Object> ans = new HashMap<>();
        ans.put("message", "User list retrieved successfully");
        ans.put("data", Map.of(
                "amount", userService.listAll().size(),
                "users", userService.listAll()
        ));
        return ResponseEntity.ok(ans);
    }

    @GetMapping("/get/info")
    public ResponseEntity<?> getUserInfo(@RequestHeader("token") String token,
                                         @RequestParam("username") String username) {

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            ans.put("message", "User info retrieved successfully");
            ans.put("data", user);
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }
}
