package com.pt.controller;

import com.pt.constant.Constants;
import com.pt.entity.User;
import com.pt.utils.JWTUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import com.pt.service.UserService;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/api/user")
@CrossOrigin(origins = "*")
public class UserController {

    @Autowired
    private UserService userService;

    @PostMapping("/register")
    public ResponseEntity<?> registerUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");
        String email = request.get("email");

        if (username == null || password == null || email == null) {
            return ResponseEntity.badRequest().body("Missing required fields");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            return ResponseEntity.badRequest().body("User already exists");
        } else {
            User newUser = new User();

            String uid = String.valueOf(System.currentTimeMillis());
            newUser.setUid(uid);
            newUser.setUsername(username);
            newUser.setPassword(password);
            newUser.setEmail(email);
            userService.save(newUser);

            Map<String, Object> ans = new HashMap<>();
            ans.put("result", "User registered successfully");
            ans.put("data", newUser);

            return ResponseEntity.ok().body(ans);
        }
    }

    @PostMapping("/login")
    public ResponseEntity<?> loginUser(@RequestParam("username") String username,
                                       @RequestParam("password") String password) {


        if (username == null || password == null) {
            return ResponseEntity.badRequest().body("Missing username or password");
        }

        User user = userService.findByUsernameAndPassword(username, password);
        Map<String, Object> ans = new HashMap<>();
        if (user != null) {
            String token = JWTUtils.generateToken(username, Constants.UserRole.USER, Constants.DEFAULT_EXPIRE_TIME);
            ans.put("result", "Login successful");
            ans.put("data", token);
            return ResponseEntity.ok().body(ans);
        } else {
            ans.put("result", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/username")
    public ResponseEntity<?> updateUsername(@RequestHeader("token") String token,
                                            @RequestParam("username") String oldUsername,
                                            @RequestParam("newUsername") String newUsername) {
        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, oldUsername, Constants.UserRole.USER)) {
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(oldUsername);
        if (user != null) {
            user.setUsername(newUsername);
            userService.save(user);
            ans.put("result", "Username updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("result", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/password")
    public ResponseEntity<?> updatePassword(@RequestHeader("token") String token,
                                            @RequestParam("username") String username,
                                            @RequestParam("newPassword") String newPassword) {
        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setPassword(newPassword);
            userService.save(user);
            ans.put("result", "Password updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("result", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/email")
    public ResponseEntity<?> updateEmail(@RequestHeader("token") String token,
                                         @RequestParam("username") String username,
                                         @RequestParam("newEmail") String newEmail) {

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setEmail(newEmail);
            userService.save(user);
            ans.put("result", "Email updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("result", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @DeleteMapping("/delete")
    public ResponseEntity<?> deleteUser(@RequestHeader("token") String token,
                                        @RequestParam("username") String username,
                                        @RequestParam("targetUsername") String targetUsername
    ) {
        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(targetUsername);
        if (user != null) {
            userService.deleteById(user.getUid());
            ans.put("result", "User deleted successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("result", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @GetMapping("/list")
    public ResponseEntity<?> listUsers(@RequestHeader("token") String token,
                                       @RequestParam("username") String username) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        Map<String, Object> ans = new HashMap<>();
        ans.put("result", "User list retrieved successfully");
        ans.put("data", Map.of(
                "amount", userService.listAll().size(),
                "users", userService.listAll()
        ));
        return ResponseEntity.ok(ans);
    }

    @GetMapping("/get/info")
    public ResponseEntity<?> getUserInfo(@RequestHeader("token") String token,
                                         @RequestParam("username") String username) {
        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("result", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            ans.put("result", "User info retrieved successfully");
            ans.put("data", user);
            return ResponseEntity.ok(ans);
        } else {
            ans.put("result", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }
}
