package com.pt.repository;

import com.pt.entity.TorrentStats;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import org.springframework.data.repository.query.Param;
import org.springframework.stereotype.Repository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

@Repository
public interface TorrentStatsRepository extends JpaRepository<TorrentStats, Long> {

    // 1. 根据种子ID查找统计信息
    Optional<TorrentStats> findByTorrentId(Long torrentId);

    // 2. 获取做种人数最多的种子（用于热门种子）
    @Query("SELECT s FROM TorrentStats s ORDER BY s.seederCount DESC")
    List<TorrentStats> findTopBySeederCount(int limit);

    // 3. 获取最近活跃的种子
    @Query("SELECT s FROM TorrentStats s ORDER BY s.lastUpdated DESC")
    List<TorrentStats> findRecentActiveTorrents(int limit);

    // 4. 根据多个种子ID批量获取统计信息
    @Query("SELECT s FROM TorrentStats s WHERE s.torrentId IN :torrentIds")
    List<TorrentStats> findByTorrentIds(@Param("torrentIds") List<Long> torrentIds);

    // 5. 获取所有需要更新的种子统计（用于定时任务）
    @Query("SELECT s.torrentId FROM TorrentStats s WHERE s.lastUpdated < :threshold")
    List<Long> findStaleStatsIds(@Param("threshold") LocalDateTime threshold);

    // 6. 增加完成次数（当有用户完成下载时调用）
    @Modifying
    @Query("UPDATE TorrentStats s SET s.completedCount = s.completedCount + 1 WHERE s.torrentId = :torrentId")
    void incrementCompletedCount(@Param("torrentId") Long torrentId);

    // 7. 更新做种者和下载者数量
    @Modifying
    @Query("UPDATE TorrentStats s SET s.seederCount = :seeders, s.leecherCount = :leechers, s.lastUpdated = CURRENT_TIMESTAMP WHERE s.torrentId = :torrentId")
    void updatePeerCounts(@Param("torrentId") Long torrentId,
                          @Param("seeders") int seeders,
                          @Param("leechers") int leechers);

    // 8. 添加缺失的方法：更新最后更新时间
    @Modifying
    @Query("UPDATE TorrentStats s SET s.lastUpdated = :lastUpdated WHERE s.torrentId = :torrentId")
    void updateLastUpdated(@Param("torrentId") Long torrentId,
                           @Param("lastUpdated") LocalDateTime lastUpdated);

    // 9. 添加缺失的方法：使用当前时间戳更新最后更新时间
    @Modifying
    @Query("UPDATE TorrentStats s SET s.lastUpdated = CURRENT_TIMESTAMP WHERE s.torrentId = :torrentId")
    void updateLastUpdatedToNow(@Param("torrentId") Long torrentId);
}