package com.pt.controller;

import com.pt.constant.Constants;
import com.pt.entity.User;
import com.pt.utils.JWTUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import com.pt.service.UserService;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/api/user")
@CrossOrigin(origins = "*")
public class UserController {

    @Autowired
    private UserService userService;

    @PostMapping("/register")
    public ResponseEntity<?> registerUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");
        String email = request.get("email");

        if (username == null || password == null || email == null) {
            return ResponseEntity.badRequest().body("Missing required fields");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            return ResponseEntity.badRequest().body("User already exists");
        } else {
            User newUser = new User();

            String uid = String.valueOf(System.currentTimeMillis());
            newUser.setUid(uid);
            newUser.setUsername(username);
            newUser.setPassword(password);
            newUser.setEmail(email);
            userService.save(newUser);

            Map<String, Object> ans = new HashMap<>();
            ans.put("success", true);
            ans.put("message", "User registered successfully");
            ans.put("data", newUser);

            return ResponseEntity.ok().body(ans);
        }
    }

    @PostMapping("/login")
    public ResponseEntity<?> loginUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");

        if (username == null || password == null) {
            return ResponseEntity.badRequest().body("Missing username or password");
        }

        User user = userService.findByUsernameAndPassword(username, password);
        Map<String, Object> ans = new HashMap<>();
        if (user != null) {
            String token = JWTUtils.generateToken(username, Constants.UserRole.USER, Constants.DEFAULT_EXPIRE_TIME);
            ans.put("success", true);
            ans.put("message", "Login successful");
            ans.put("data", Map.of(
                "token", token,
                "user", user
            ));
            return ResponseEntity.ok().body(ans);
        } else {
            ans.put("success", false);
            ans.put("message", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/username")
    public ResponseEntity<?> updateUsername(@RequestHeader("token") String token,
                                            @RequestParam("username") String oldUsername,
                                            @RequestParam("newUsername") String newUsername) {
        if(!JWTUtils.checkToken(token, oldUsername, Constants.UserRole.USER)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        User user = userService.findByUsername(oldUsername);
        if (user != null) {
            user.setUsername(newUsername);
            userService.save(user);
            return ResponseEntity.ok("Username updated successfully");
        } else {
            return ResponseEntity.badRequest().body("User not found");
        }
    }

    @PostMapping("/update/password")
    public ResponseEntity<?> updatePassword(@RequestHeader("token") String token,
                                            @RequestParam("username") String username,
                                            @RequestParam("newPassword") String newPassword) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setPassword(newPassword);
            userService.save(user);
            return ResponseEntity.ok("Password updated successfully");
        } else {
            return ResponseEntity.badRequest().body("Invalid username or password");
        }
    }

    @PostMapping("/update/email")
    public ResponseEntity<?> updateEmail(@RequestHeader("token") String token,
                                         @RequestParam("username") String username,
                                         @RequestParam("newEmail") String newEmail) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setEmail(newEmail);
            userService.save(user);
            return ResponseEntity.ok("Email updated successfully");
        } else {
            return ResponseEntity.badRequest().body("User not found");
        }
    }

    @DeleteMapping("/delete")
    public ResponseEntity<?> deleteUser(@RequestHeader("token") String token,
                                        @RequestParam("username") String username,
                                        @RequestParam("targetUsername") String targetUsername
    ) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        User user = userService.findByUsername(targetUsername);
        if (user != null) {
            userService.deleteById(user.getUid());
            return ResponseEntity.ok("User deleted successfully");
        } else {
            return ResponseEntity.badRequest().body("User not found");
        }
    }

    @GetMapping("/list")
    public ResponseEntity<?> listUsers(@RequestHeader("token") String token,
                                       @RequestParam("username") String username) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        Map<String, Object> ans = new HashMap<>();
        ans.put("result", "User list retrieved successfully");
        ans.put("amount", userService.listAll().size());
        ans.put("users", userService.listAll());
        return ResponseEntity.ok(ans);
    }

    @GetMapping("/get/info")
    public ResponseEntity<?> getUserInfo(@RequestHeader("token") String token,
                                         @RequestParam("username") String username) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            Map<String, Object> ans = new HashMap<>();
            ans.put("result", "User info retrieved successfully");
            ans.put("user", user);
            return ResponseEntity.ok(ans);
        } else {
            return ResponseEntity.badRequest().body("User not found");
        }
    }
}
