package com.pt.controller;

import com.pt.Item.CommentInfo;
import com.pt.constant.Constants;
import com.pt.entity.Post;
import com.pt.entity.User;
import com.pt.service.CommentService;
import com.pt.service.PostService;
import com.pt.service.UserService;
import com.pt.utils.JWTUtils;
import com.pt.entity.Comment;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/comment")
@CrossOrigin(origins = "*")
public class CommentController {

    @Autowired
    private CommentService commentService;

    @Autowired
    private PostService postService;

    @Autowired
    private UserService userService;

    @PostMapping("/add")
    public ResponseEntity<?> addComment(
            @RequestHeader("token") String token,
            @RequestBody Map<String, String> request
    ) {
        String content = request.get("content");
        String username = request.get("username");
        int postId = Integer.parseInt(request.get("postId"));
        Integer reviewer = request.get("reviewer") != null ? Integer.parseInt(request.get("reviewer")) : null;

        Map<String, Object> ans = new HashMap<>();

        if (!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        commentService.addComment(content, username, postId, reviewer);
        ans.put("message", "Comment added successfully");
        return ResponseEntity.ok(ans);
    }

    @DeleteMapping("/delete")
    public ResponseEntity<?> deleteComment(
            @RequestHeader("token") String token,
            @RequestParam("username") String username,
            @RequestParam("commentId") int commentId
    ) {

        Map<String, Object> ans = new HashMap<>();

        if (!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        commentService.deleteComment(commentId);
        ans.put("message", "Comment deleted successfully");
        return ResponseEntity.ok(ans);
    }

    @GetMapping("/get")
    public ResponseEntity<?> getComments(
            @RequestHeader("token") String token,
            @RequestParam("username") String username,
            @RequestParam("postId") int postId
    ) {
        Map<String, Object> ans = new HashMap<>();

        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        Post post = postService.findPostById(postId);
        List<Comment> comments = commentService.getCommentsByPostId(postId);
        List<CommentInfo> commentInfos = new ArrayList<>();
        for(Comment comment : comments) {
            CommentInfo commentInfo = new CommentInfo();
            commentInfo.setCommentId(comment.getCommentId());
            commentInfo.setContent(comment.getContent());
            commentInfo.setWriter(comment.getWriter());
            commentInfo.setPublishDate(comment.getPublishDate());

            if(comment.getReviewer() != null) {
                Comment r = commentService.getCommentById(comment.getReviewer());
                if(r != null) {
                    commentInfo.setReviewer(r.getWriter());
                    commentInfo.setReviewerId(comment.getReviewer());
                } else {
                    commentInfo.setReviewer("Unknown Reviewer");
                    commentInfo.setReviewerId(0);
                }
            } else {
                commentInfo.setReviewer("");
                commentInfo.setReviewerId(0);
            }

            commentInfos.add(commentInfo);
        }

        ans.put("message", "Comments retrieved successfully");
        ans.put("data", Map.of(
                "content", post.getContent(),
                "comments", commentInfos
        ));
        return ResponseEntity.ok(ans);
    }
}
