package com.pt.controller;

import com.pt.constant.Constants;
import com.pt.entity.User;
import com.pt.service.InvitedCodeService;
import com.pt.utils.JWTUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import com.pt.service.UserService;

import java.util.HashMap;
import java.util.Map;

@RestController
@RequestMapping("/api/user")
@CrossOrigin(origins = "*")
public class UserController {

    @Autowired
    private UserService userService;
    private InvitedCodeService invitedCodeService;

    @PostMapping("/register")
    public ResponseEntity<?> registerUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");
        String email = request.get("email");

        if (username == null || password == null || email == null) {
            return ResponseEntity.badRequest().body("Missing required fields");
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            return ResponseEntity.badRequest().body("User already exists");
        } else {
            User newUser = new User();

            if(request.get("invitedCode") != null) {
                String invitedCode = request.get("invitedCode");
                if (!invitedCodeService.isCodeValid(invitedCode)) {
                    return ResponseEntity.badRequest().body("Invalid invited code");
                }
                else {
                    newUser.setLevel(Constants.UserLevel.FRESH.getValue());
                    newUser.setPoints(300);
                }
            }

            String uid = String.valueOf(System.currentTimeMillis());
            newUser.setUid(uid);
            newUser.setUsername(username);
            newUser.setPassword(password);
            newUser.setEmail(email);
            userService.save(newUser);

            Map<String, Object> ans = new HashMap<>();
            ans.put("message", "User registered successfully");
            ans.put("data", newUser);

            return ResponseEntity.ok().body(ans);
        }
    }

    @PostMapping("/login")
    public ResponseEntity<?> loginUser(@RequestBody Map<String, String> request) {
        String username = request.get("username");
        String password = request.get("password");

        if (username == null || password == null) {
            return ResponseEntity.badRequest().body("Missing username or password");
        }

        User user = userService.findByUsernameAndPassword(username, password);
        Map<String, Object> ans = new HashMap<>();
        if (user != null) {
            String token = JWTUtils.generateToken(username, Constants.UserRole.USER, Constants.DEFAULT_EXPIRE_TIME);
            ans.put("message", "Login successful");
            ans.put("data", Map.of(
                    "token", token,
                    "uid", user.getUid(),
                    "username", user.getUsername(),
                    "email", user.getEmail(),
                    "level", user.getLevel(),
                    "points", user.getPoints(),
                    "uploaded", user.getUploaded(),
                    "downloaded", user.getDownloaded(),
                    "shareRatio", user.getShareRatio()
            ));
            return ResponseEntity.ok().body(ans);
        } else {
            ans.put("message", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/username")
    public ResponseEntity<?> updateUsername(@RequestHeader("token") String token,
                                            @RequestBody Map<String, String> request) {
        String oldUsername = request.get("username");
        String newUsername = request.get("newUsername");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, oldUsername, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(oldUsername);
        if (user != null) {
            user.setUsername(newUsername);
            userService.save(user);
            ans.put("message", "Username updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/password")
    public ResponseEntity<?> updatePassword(@RequestHeader("token") String token,
                                            @RequestBody Map<String, String> request
                                            ) {
        String username = request.get("username");
        String newPassword = request.get("newPassword");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setPassword(newPassword);
            userService.save(user);
            ans.put("message", "Password updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "Invalid username or password");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @PostMapping("/update/email")
    public ResponseEntity<?> updateEmail(@RequestHeader("token") String token,
                                         @RequestBody Map<String, String> request) {
        String username = request.get("username");
        String newEmail = request.get("newEmail");

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            user.setEmail(newEmail);
            userService.save(user);
            ans.put("message", "Email updated successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @DeleteMapping("/delete")
    public ResponseEntity<?> deleteUser(@RequestHeader("token") String token,
                                        @RequestParam("username") String username,
                                        @RequestParam("targetUsername") String targetUsername) {

        Map<String, Object> ans = new HashMap<>();
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(targetUsername);
        if (user != null) {
            userService.deleteById(user.getUid());
            ans.put("message", "User deleted successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    @GetMapping("/list")
    public ResponseEntity<?> listUsers(@RequestHeader("token") String token,
                                       @RequestParam("username") String username) {
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.ADMIN)) {
            return ResponseEntity.badRequest().body("Invalid token");
        }

        Map<String, Object> ans = new HashMap<>();
        ans.put("message", "User list retrieved successfully");
        ans.put("data", Map.of(
                "amount", userService.listAll().size(),
                "users", userService.listAll()
        ));
        return ResponseEntity.ok(ans);
    }

    @GetMapping("/get/info")
    public ResponseEntity<?> getUserInfo(@RequestHeader("token") String token,
                                         @RequestParam("username") String username) {

        Map<String, Object> ans = new HashMap<>();
        System.out.println("Enter user info ");
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            System.out.println("Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user != null) {
            ans.put("message", "User info retrieved successfully");
            ans.put("data", user);
            System.out.println("User info retrieved successfully");
            return ResponseEntity.ok(ans);
        } else {
            ans.put("message", "User not found");
            System.out.println("User not found");
            return ResponseEntity.badRequest().body(ans);
        }
    }

    /**
     * 获取用户统计信息
     */
    @GetMapping("/stats/{username}")
    public ResponseEntity<?> getUserStats(
            @RequestHeader("token") String token,
            @PathVariable String username) {
        
        Map<String, Object> ans = new HashMap<>();
        
        if(!JWTUtils.checkToken(token, username, Constants.UserRole.USER)) {
            ans.put("message", "Invalid token");
            return ResponseEntity.badRequest().body(ans);
        }

        User user = userService.findByUsername(username);
        if (user == null) {
            ans.put("message", "User not found");
            return ResponseEntity.badRequest().body(ans);
        }

        // 计算分享率
        double ratio = user.getDownloaded() == 0 ? 
            (user.getUploaded() > 0 ? Double.MAX_VALUE : 0) : 
            (double) user.getUploaded() / user.getDownloaded();
        
        // 格式化分享率为两位小数
        ratio = Math.round(ratio * 100.0) / 100.0;

        // 构建返回数据
        Map<String, Object> stats = new HashMap<>();
        double uploadSize = user.getUploaded() / (1024.0 * 1024.0 * 1024.0);
        double downloadSize = user.getDownloaded() / (1024.0 * 1024.0 * 1024.0);
        
        stats.put("uploadSize", uploadSize); // 转换为GB
        stats.put("downloadSize", downloadSize); // 转换为GB
        stats.put("ratio", ratio);
        stats.put("points", user.getPoints());
        stats.put("userClass", getUserClass(user.getLevel()));
        stats.put("level", user.getLevel());

        ans.put("message", "User stats retrieved successfully");
        ans.put("data", stats);
        return ResponseEntity.ok(ans);
    }

    /**
     * 根据用户等级返回对应的用户类别名称
     */
    private String getUserClass(int level) {
        switch (level) {
            case 5: return "大佬";
            case 4: return "专家";
            case 3: return "熟练";
            case 2: return "入门";
            default: return "新用户";
        }
    }
}
