package com.pt.scheduler;

import com.pt.entity.PeerInfoEntity;
import com.pt.entity.User;
import com.pt.repository.PeerInfoRepository;
import com.pt.service.UserService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 定时任务，用于同步用户的流量数据
 */
@Component
public class UserTrafficSyncScheduler {

    private static final Logger logger = LoggerFactory.getLogger(UserTrafficSyncScheduler.class);

    @Autowired
    private UserService userService;

    @Autowired
    private PeerInfoRepository peerInfoRepository;

    /**
     * 每小时执行一次流量同步
     * 确保用户的总流量数据准确
     */
    @Scheduled(cron = "0 0 * * * ?")
    @Transactional
    public void syncUserTraffic() {
        logger.info("开始执行定时任务：同步用户流量数据...");
        try {
            List<User> users = userService.listAll();
            for (User user : users) {
                // 获取用户所有peer的总流量
                Long totalUploaded = peerInfoRepository.sumUploadedByUsername();
                Long totalDownloaded = peerInfoRepository.sumDownloadedByUsername();

                // 如果查询结果为空，设置为0
                totalUploaded = totalUploaded != null ? totalUploaded : 0L;
                totalDownloaded = totalDownloaded != null ? totalDownloaded : 0L;

                // 更新用户流量
                user.setUploaded(totalUploaded);
                user.setDownloaded(totalDownloaded);
                userService.save(user);

                // 更新用户等级
                userService.updateUserLevel(user.getUid());

                logger.info("用户 {} 流量同步完成: 上传={}, 下载={}", 
                    user.getUsername(), totalUploaded, totalDownloaded);
            }
            logger.info("定时任务完成：所有用户流量数据已同步");
        } catch (Exception e) {
            logger.error("执行用户流量同步定时任务时发生错误", e);
        }
    }
} 