package com.turn.ttorrent.client;

import com.turn.ttorrent.common.AnnounceableInformation;
import com.turn.ttorrent.common.Pair;
import com.turn.ttorrent.common.TorrentUtils;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

// 客户端获取
//负责管理两类 torrent：活跃的（正在进行下载或上传的）和已加载的（已经被解析并加载进内存但可能未被活跃使用的） torrent。
//该类提供了方法来添加、获取、移除、检查这些 torrent，同时支持高效的并发访问，确保线程安全。
public class TorrentsStorage {

  private final ReadWriteLock readWriteLock;
  private final Map<String, SharedTorrent> activeTorrents;
  private final Map<String, LoadedTorrent> loadedTorrents;

  public TorrentsStorage() {
    readWriteLock = new ReentrantReadWriteLock();
    activeTorrents = new HashMap<String, SharedTorrent>();
    loadedTorrents = new HashMap<String, LoadedTorrent>();
  }

  //根据hash查找是否有对应种子
  public boolean hasTorrent(String hash) {
    try {
      readWriteLock.readLock().lock();
      return loadedTorrents.containsKey(hash);
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  //获取已加载的torrent
  public LoadedTorrent getLoadedTorrent(String hash) {
    try {
      readWriteLock.readLock().lock();
      return loadedTorrents.get(hash);
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  // 处理peer断开连接的情况 种子不活跃了在此处进行操作
  public void peerDisconnected(String torrentHash) {
    final SharedTorrent torrent;
    try {
      readWriteLock.writeLock().lock();
      torrent = activeTorrents.get(torrentHash);
      if (torrent == null) return;

      boolean isTorrentFinished = torrent.isFinished();
      if (torrent.getDownloadersCount() == 0 && isTorrentFinished) {
        activeTorrents.remove(torrentHash);
      } else {
        return;
      }
    } finally {
      readWriteLock.writeLock().unlock();
    }
    torrent.close();
  }

  //获取活跃的torrent
  public SharedTorrent getTorrent(String hash) {
    try {
      readWriteLock.readLock().lock();
      return activeTorrents.get(hash);
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  // 将已经加载的种子加入
  public void addTorrent(String hash, LoadedTorrent torrent) {
    try {
      readWriteLock.writeLock().lock();
      loadedTorrents.put(hash, torrent);
    } finally {
      readWriteLock.writeLock().unlock();
    }
  }

  public SharedTorrent putIfAbsentActiveTorrent(String hash, SharedTorrent torrent) {
    try {
      readWriteLock.writeLock().lock();
      final SharedTorrent old = activeTorrents.get(hash);
      if (old != null) return old;

      return activeTorrents.put(hash, torrent);
    } finally {
      readWriteLock.writeLock().unlock();
    }
  }

  public Pair<SharedTorrent, LoadedTorrent> remove(String hash) {
    final Pair<SharedTorrent, LoadedTorrent> result;
    try {
      readWriteLock.writeLock().lock();
      final SharedTorrent sharedTorrent = activeTorrents.remove(hash);
      final LoadedTorrent loadedTorrent = loadedTorrents.remove(hash);
      result = new Pair<SharedTorrent, LoadedTorrent>(sharedTorrent, loadedTorrent);
    } finally {
      readWriteLock.writeLock().unlock();
    }
    if (result.second() != null) {
      try {
        result.second().getPieceStorage().close();
      } catch (IOException ignored) {
      }
    }
    if (result.first() != null) {
      result.first().closeFully();
    }
    return result;
  }

  // 获取活跃的种子
  public List<SharedTorrent> activeTorrents() {
    try {
      readWriteLock.readLock().lock();
      return new ArrayList<SharedTorrent>(activeTorrents.values());
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  public List<AnnounceableInformation> announceableTorrents() {
    List<AnnounceableInformation> result = new ArrayList<AnnounceableInformation>();
    try {
      readWriteLock.readLock().lock();
      for (LoadedTorrent loadedTorrent : loadedTorrents.values()) {
        AnnounceableInformation announceableInformation = loadedTorrent.createAnnounceableInformation();
        if (TorrentUtils.isTrackerLessInfo(announceableInformation)) continue;
        result.add(announceableInformation);
      }
      return result;
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  public List<LoadedTorrent> getLoadedTorrents() {
    try {
      readWriteLock.readLock().lock();
      return new ArrayList<LoadedTorrent>(loadedTorrents.values());
    } finally {
      readWriteLock.readLock().unlock();
    }
  }

  public void clear() {
    final Collection<SharedTorrent> sharedTorrents;
    final Collection<LoadedTorrent> loadedTorrents;
    try {
      readWriteLock.writeLock().lock();
      sharedTorrents = new ArrayList<SharedTorrent>(activeTorrents.values());
      loadedTorrents = new ArrayList<LoadedTorrent>(this.loadedTorrents.values());
      this.loadedTorrents.clear();
      activeTorrents.clear();
    } finally {
      readWriteLock.writeLock().unlock();
    }
    for (SharedTorrent sharedTorrent : sharedTorrents) {
      sharedTorrent.closeFully();
    }
    for (LoadedTorrent loadedTorrent : loadedTorrents) {
      try {
        loadedTorrent.getPieceStorage().close();
      } catch (IOException ignored) {
      }
    }
  }
}
