package com.ptp.ptplatform.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ptp.ptplatform.entity.HelpComment;
import com.ptp.ptplatform.service.HelpCommentService;
import com.ptp.ptplatform.service.HelpPostService;
import com.ptp.ptplatform.utils.Result;
import lombok.AllArgsConstructor;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.List;
import java.util.UUID;

@RestController
@RequestMapping("/help/comments")
@AllArgsConstructor
public class HelpCommentController {
    private final HelpCommentService commentService;
    private final HelpPostService postService;

    @PostMapping("/{commentId}/like")
    public Result like(@PathVariable int commentId) {
        commentService.incrementLike(commentId);
        return Result.ok();
    }

    // 获取评论的回复
    @GetMapping("/{commentId}/replies")
    public Result getReplies(@PathVariable int commentId) {
        List<HelpComment> replies = commentService.getReplies(commentId);
        return Result.ok().data("replies", replies);
    }

    @PostMapping(value = "/{commentId}/replies", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public Result addReply(@PathVariable int commentId,
                           @RequestParam("authorId") String authorId,
                           @RequestParam("content") String content,
                           @RequestParam(value = "image", required = false) MultipartFile image) {
        // 校验父评论是否存在
        HelpComment parentComment = commentService.getById(commentId);
        if (parentComment == null) {
            return Result.error(404).setMessage("被回复的评论不存在");
        }

        // 构造回复评论对象
        HelpComment reply = new HelpComment();
        reply.setPostId(parentComment.getPostId());
        reply.setParentId(commentId);
        reply.setAuthorId(authorId);              // String 类型
        reply.setReplyTo(parentComment.getAuthorId());
        reply.setContent(content);
        reply.setCreateTime(LocalDateTime.now());
        reply.setLikeCount(0);

        // 可选：处理图片上传
        if (image != null && !image.isEmpty()) {
            try {
                String fileExt = image.getOriginalFilename()
                        .substring(image.getOriginalFilename().lastIndexOf('.') + 1);
                String fileName = UUID.randomUUID() + "." + fileExt;

                String uploadDir = System.getProperty("user.dir") + File.separator + "uploads";
                File dir = new File(uploadDir);
                if (!dir.exists()) dir.mkdirs();

                File dest = new File(dir, fileName);
                image.transferTo(dest);

                // 保存相对访问路径
                reply.setImageUrl("/uploads/" + fileName);
            } catch (IOException e) {
                return Result.error(404).setMessage("图片上传失败：" + e.getMessage());
            }
        }

        // 保存到数据库
        commentService.save(reply);

        // 更新帖子评论数
        postService.incrementReplyCount(parentComment.getPostId());

        // 返回新回复及其父评论 ID
        return Result.ok()
                .data("reply", reply)
                .data("parentCommentId", commentId);
    }


    // 删除评论
    @DeleteMapping("/{commentId}")
    public Result deleteComment(
            @PathVariable int commentId,
            @RequestParam String authorId) {

        // 1. 获取要删除的评论
        HelpComment comment = commentService.getById(commentId);
        if (comment == null) {
            return Result.error(404).setMessage("评论不存在");
        }

        // 2. 验证当前用户是否有权限删除（比对authorId）
        if (!comment.getAuthorId().equals(authorId)) {
            return Result.error(403).setMessage("无权删除此评论");
        }

        // 3. 递归删除所有子回复
        deleteRepliesRecursively(commentId);

        // 4. 删除主评论
        boolean removed = commentService.removeById(commentId);

//            // 5. 更新帖子回复数（如果需要）
//            if (removed && comment.getParentId() == 0) {
//                postService.decrementReplyCount(comment.getPostId());
//            }

        return removed ? Result.ok() : Result.error(500).setMessage("删除评论失败");
    }
    // 递归删除所有回复
    private void deleteRepliesRecursively(int parentCommentId) {
        // 添加保护措施防止无限递归
        if (parentCommentId <= 0) return;

        // 查找所有直接回复
        List<HelpComment> replies = commentService.list(
                new QueryWrapper<HelpComment>().eq("parent_id", parentCommentId)
        );

        // 递归删除每个回复及其子回复
        for (HelpComment reply : replies) {
            deleteRepliesRecursively(reply.getId());
            commentService.removeById(reply.getId());

//                // 如果是主评论（parent_id=0），更新帖子回复数
//                if (reply.getParentId() == 0) {
//                    postService.decrementReplyCount(reply.getPostId());
//                }
        }
    }


}