package com.ptp.ptplatform.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ptp.ptplatform.entity.HelpComment;
import com.ptp.ptplatform.entity.HelpPost;
import com.ptp.ptplatform.service.HelpCommentService;
import com.ptp.ptplatform.service.HelpPostService;
import com.ptp.ptplatform.service.NotificationService;
import com.ptp.ptplatform.utils.Result;
import lombok.AllArgsConstructor;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.time.LocalDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@RestController
@RequestMapping("/help/posts")
@AllArgsConstructor
public class HelpPostController {
    private final HelpPostService postService;
    private final HelpCommentService commentService;
    private final NotificationService notificationService;

    // 修改创建帖子的方法，支持图片上传
    @PostMapping(consumes = MediaType.MULTIPART_FORM_DATA_VALUE, produces = MediaType.APPLICATION_JSON_VALUE)
    public Result createPost(

            @RequestParam("title") String title,
            @RequestParam("content") String content,
            @RequestParam("authorId") String authorId,
            @RequestParam(value = "image", required = false) MultipartFile image) {

        HelpPost post = new HelpPost();
        post.setTitle(title);
        post.setContent(content);
        post.setAuthorId(authorId);
        post.setCreateTime(LocalDateTime.now());
        post.setLikeCount(0);
        post.setReplyCount(0);
        post.setImageUrl(null);
//        post.setIsSolved(false);

        // 处理图片上传
        if (image != null && !image.isEmpty()) {
            try {
                String originalFilename = image.getOriginalFilename();
                if(originalFilename != null) {
                    String fileExt = originalFilename.substring(image.getOriginalFilename().lastIndexOf('.') + 1);
                    String fileName = UUID.randomUUID() + "." + fileExt;

                    String uploadDir = System.getProperty("user.dir") + File.separator + "uploads";
                    File dir = new File(uploadDir);
                    if (!dir.exists()) dir.mkdirs();

                    File dest = new File(dir, fileName);
                    image.transferTo(dest);

                    // 保存相对访问路径
                    post.setImageUrl("/uploads/" + fileName);
                }
            } catch (IOException e) {
                return Result.error(404).setMessage("图片上传失败：" + e.getMessage());
            }
        } else {
            post.setImageUrl(null);
        }
        postService.save(post);
        return Result.ok().data("post", post);
    }

    // 列表分页
    @GetMapping
    public Result listPosts(@RequestParam(defaultValue = "1") int page,
                            @RequestParam(defaultValue = "5") int size) {
        IPage<HelpPost> ipage = postService.page(
                new Page<>(page, size),
                new QueryWrapper<HelpPost>().orderByDesc("create_time")
        );
        return Result.ok()
                .data("records", ipage.getRecords())
                .data("total", ipage.getTotal());
    }

    @GetMapping("/search")
    public Result searchPosts(
            @RequestParam String keyword,
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "5") int size) {

        // 1. 把搜索词拆分成单词（中英文混合拆分）
        List<String> keywords = splitKeywords(keyword); // 例如 "你好hello" → ["你", "好", "hello"]

        // 2. 构建动态查询条件（每个关键词用 OR 连接）
        QueryWrapper<HelpPost> queryWrapper = new QueryWrapper<>();
        for (String word : keywords) {
            queryWrapper.and(wrapper ->
                    // 使用LOWER函数使搜索不区分大小写
                    wrapper.apply("LOWER(title) LIKE LOWER({0})", "%" + word + "%")
                            .or()
                            .apply("LOWER(content) LIKE LOWER({0})", "%" + word + "%")
            );
        }
        queryWrapper.orderByDesc("create_time");

        // 3. 执行分页查询
        IPage<HelpPost> ipage = postService.page(new Page<>(page, size), queryWrapper);
        return Result.ok()
                .data("records", ipage.getRecords())
                .data("total", ipage.getTotal());
    }

    /**
     * 拆分关键词（支持中英文混合）
     * 例如："你好hello 世界" → ["你", "好", "hello", "世", "界"]
     */
    private List<String> splitKeywords(String keyword) {
        List<String> keywords = new ArrayList<>();

        // 用正则匹配中文字符和英文单词
        Pattern pattern = Pattern.compile("([\\u4e00-\\u9fa5])|([a-zA-Z0-9]+)");
        Matcher matcher = pattern.matcher(keyword);

        while (matcher.find()) {
            String matched = matcher.group();
            keywords.add(matched);
        }

        return keywords;
    }

    @GetMapping("/{Id}")
    public Result getPost(@PathVariable int Id) {
        HelpPost post = postService.getById(Id);
        if (post == null) {
            return Result.error(404).setMessage("帖子不存在");
        }

        // 获取所有评论（按创建时间排序）
        List<HelpComment> allComments = commentService.list(
                new QueryWrapper<HelpComment>()
                        .eq("post_id", Id)
                        .orderByAsc("create_time")
        );

        // 获取实时评论总数（包括所有层级）
        int totalComments = allComments.size(); // 直接使用列表大小
        post.setReplyCount(totalComments); // 更新计数

        // 构建评论树形结构
        List<HelpComment> rootComments = new ArrayList<>();
        Map<Integer, HelpComment> commentMap = new HashMap<>();

        // 第一遍：初始化所有评论到map中
        for (HelpComment comment : allComments) {
            comment.setReplies(new ArrayList<>()); // 初始化replies列表
            commentMap.put(comment.getId(), comment);
        }

        // 第二遍：构建父子关系
        for (HelpComment comment : allComments) {
            if (comment.getParentId() == 0) {
                rootComments.add(comment);
            } else {
                HelpComment parent = commentMap.get(comment.getParentId());
                if (parent != null) {
                    parent.getReplies().add(comment);
                }
            }
        }

        return Result.ok()
                .data("post", post)
                .data("comments", rootComments);
    }

    // 点赞帖子
    @PostMapping("/{Id}/like")
    public Result likePost(
            @PathVariable("Id") int Id,
            @RequestParam("likerId") String likerId
    ) {
        // 调用 ServiceImpl.likePost(...)，自动更新 like_count 并插入通知
        postService.likePost(Id, likerId);
        return Result.ok().message("点赞成功");
    }

    // 评论帖子
    @PostMapping(value = "/{Id}/comments", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public Result commentOnPost(
            @PathVariable("Id") int Id,
            @RequestParam("authorId") String authorId,
            @RequestParam("content") String content,
            @RequestParam(value = "image", required = false) MultipartFile image
    ) {
        // 1. 构造 HelpComment 对象
        HelpComment comment = new HelpComment();
        comment.setPostId(Id);
        comment.setAuthorId(authorId);
        comment.setContent(content);
        comment.setCreateTime(LocalDateTime.now());
        comment.setLikeCount(0);
        comment.setParentId(0);       // 这里是对帖子的一级评论
        comment.setImageUrl(null);    // 默认无图片

        // 2. 处理可选的图片上传逻辑
        if (image != null && !image.isEmpty()) {
            try {
                String fileExt = image.getOriginalFilename()
                        .substring(image.getOriginalFilename().lastIndexOf('.') + 1);
                String fileName = UUID.randomUUID().toString() + "." + fileExt;

                // 上传到项目根目录的 uploads 文件夹
                String uploadDir = System.getProperty("user.dir")
                        + File.separator + "uploads";
                File dir = new File(uploadDir);
                if (!dir.exists()) dir.mkdirs();

                File dest = new File(dir, fileName);
                image.transferTo(dest);

                comment.setImageUrl("/uploads/" + fileName);
            } catch (IOException e) {
                return Result.error(500).message("图片上传失败：" + e.getMessage());
            }
        }

        // 3. 通过 ServiceImpl 完成“保存评论 + 更新 reply_count + 发通知给帖子作者”
        commentService.commentOnHelpPost(comment);

        // 4. 构造返回值：把刚刚保存的评论（从数据库中查出来的完整对象）和帖子最新的 replyCount 都返回
        //    注意：comment.insert(...) 后，comment.getId() 将有 DB 自动生成的主键
        HelpComment newComment = commentService.getById(comment.getId());
        int updatedReplyCount = postService.getById(Id).getReplyCount();

        return Result.ok()
                .data("comment", newComment)
                .data("newReplyCount", updatedReplyCount);
    }

    // 删除帖子
    @DeleteMapping("/{postId}")
    public Result deletePost(@PathVariable int postId,
                             @RequestParam String authorId) {
        try {
            // 1. 验证当前用户是否有权限删除（比对authorId）
            HelpPost post = postService.getById(postId);
            if (post == null) {
                return Result.error(404).setMessage("帖子不存在");
            }
            if (!post.getAuthorId().equals(authorId)) {
                return Result.error(403).setMessage("无权删除此帖子");
            }

            // 2. 删除帖子及关联评论
            commentService.remove(new QueryWrapper<HelpComment>().eq("post_id", postId));
            boolean removed = postService.removeById(postId);
            return removed ? Result.ok() : Result.error(404).setMessage("删除评论失败");
        } catch (Exception e) {
            return Result.error(500).setMessage("服务器错误: " + e.getMessage());
        }
    }
}