package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.Notification;
import com.ptp.ptplatform.entity.USER;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.service.NotificationService;
import com.ptp.ptplatform.utils.Result;
import lombok.AllArgsConstructor;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDateTime;
import java.util.List;

/**
 * 调用示例：
 *   POST /api/system/announce
 *   Content-Type: application/json
 *   {
 *     "title":   "系统维护提醒",
 *     "content": "本周六（6月5日）凌晨2:00-4:00将进行全站维护，届时功能暂不可用。"
 *   }
 */
@RestController
@RequestMapping("/system")
@AllArgsConstructor
public class SystemNotificationController {

    private final UserMapper userMapper;                  // 用于查询所有用户
    private final NotificationService notificationService; // 用于保存通知


    //发布一条系统公告，系统会给当前所有注册用户发一条通知。

    @PostMapping("/announce")
    public Result announce(@RequestBody AnnounceRequest body) {
        // 1. 参数校验
        if (body.getTitle() == null || body.getTitle().isBlank()
                || body.getContent() == null || body.getContent().isBlank()) {
            return Result.error(400).message("请求参数 title/content 不能为空");
        }

        // 2. 查询所有用户
        List<USER> allUsers = userMapper.selectAllUsers();
        if (allUsers == null || allUsers.isEmpty()) {
            return Result.error(500).message("当前系统中没有可发送通知的用户");
        }

        // 3. 遍历所有用户，为每个用户创建并保存一条 Notification
        int sentCount = 0;
        LocalDateTime now = LocalDateTime.now();
        for (USER u : allUsers) {
            String uid = u.getUsername();  // 获取用户名（用户主键）

            Notification n = new Notification();
            n.setUserId(uid);                 // 通知接收者 = 当前用户名
            n.setType("SYSTEM");              // 通知类型标记为系统消息
            n.setTitle(body.getTitle());      // 通知标题
            n.setContent(body.getContent());  // 通知内容
            n.setTargetId(null);              // 系统公告通常无需跳转，设为 null
            n.setTargetType(null);
            n.setIsRead(false);
            n.setCreateTime(now);

            notificationService.saveNotification(n);
            sentCount++;
        }

        // 4. 返回发送成功及发送数量
        return Result.ok()
                .message("公告已发送，通知条数：" + sentCount)
                .data("sentCount", sentCount);
    }

    public static class AnnounceRequest {
        private String title;
        private String content;

        // Getter & Setter
        public String getTitle() {
            return title;
        }
        public void setTitle(String title) {
            this.title = title;
        }
        public String getContent() {
            return content;
        }
        public void setContent(String content) {
            this.content = content;
        }
    }
}
