package com.ptp.ptplatform.controller;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ptp.ptplatform.entity.*;
import com.ptp.ptplatform.mapper.DiscountMapper;
import com.ptp.ptplatform.mapper.TorrentMapper;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.mapper.DownloadTorrentMapper;
import com.ptp.ptplatform.service.*;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ptp.ptplatform.utils.Result;
import com.ptp.ptplatform.utils.SizeCalculation;
import com.turn.ttorrent.bcodec.BDecoder;
import com.turn.ttorrent.bcodec.BEValue;
import com.turn.ttorrent.tracker.TrackedPeer;
import com.turn.ttorrent.tracker.TrackedTorrent;
import jakarta.annotation.PostConstruct;
import jakarta.annotation.Resource;
import jakarta.servlet.http.HttpServletRequest;
import lombok.AllArgsConstructor;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.web.multipart.MultipartFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.UnknownHostException;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;


//实现种子的增删改查
@RestController
@RequestMapping("/torrent")
@AllArgsConstructor
public class TorrentController {
    private final TorrentService postService;
    private final TorrentCommentService commentService;
    private final NotificationService notificationService;

    @Resource
    private TorrentMapper torrentMapper;
    @Resource
    private UserMapper userMapper;
    @Resource
    private DiscountMapper discountMapper;
    @Resource
    private DownloadTorrentMapper downloadTorrentMapper;

    @Resource
    private UserController userController;

    private TrackerService trackerservice = new TrackerService(new ObjectMapper());
    private ClientService clientservice = new ClientService();
    private static final Logger logger = LoggerFactory.getLogger(TorrentController.class);

    @GetMapping
    public Result listPosts(
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "5") int size,
            @RequestParam(required = false) String category,
            @RequestParam(required = false) String subtitle,
            @RequestParam(required = false) String region,
            @RequestParam(required = false) String resolution) {

        // 构建查询条件
        QueryWrapper<TORRENT> queryWrapper = new QueryWrapper<>();
        queryWrapper.orderByDesc("create_time");

        if (category != null && !category.isEmpty() && !"all".equals(category)) {
            queryWrapper.eq("category", category);
        }
        if (subtitle != null && !subtitle.isEmpty() && !"all".equals(subtitle)) {
            queryWrapper.eq("subtitle", subtitle);
        }
        if (region != null && !region.isEmpty() && !"all".equals(region)) {
            queryWrapper.eq("region", region);
        }
        if (resolution != null && !resolution.isEmpty() && !"all".equals(resolution)) {
            queryWrapper.eq("resolution", resolution);
        }

        // 执行分页查询
        IPage<TORRENT> ipage = new Page<>(page, size);
        IPage<TORRENT> resultPage = torrentMapper.selectPage(ipage, queryWrapper);

        // 转换size字段为格式化字符串
        List<Map<String, Object>> formattedRecords = resultPage.getRecords().stream()
                .map(torrent -> {
                    Map<String, Object> map = new HashMap<>();
                    map.put("id", torrent.getId());
                    map.put("torrentName", torrent.getTorrentName());
                    map.put("description", torrent.getDescription());
                    map.put("category", torrent.getCategory());
                    map.put("region", torrent.getRegion());
                    map.put("resolution", torrent.getResolution());
                    map.put("subtitle", torrent.getSubtitle());
                    map.put("size", formatFileSize(torrent.getSize())); // 格式化size
                    map.put("hash", torrent.getHash());
                    map.put("username", torrent.getUsername());
                    map.put("filePath", torrent.getFilePath());
                    map.put("like_count", torrent.getLike_count());
                    map.put("reply_count", torrent.getReply_count());
                    map.put("createTime", torrent.getCreateTime());
                    return map;
                })
                .collect(Collectors.toList());

        return Result.ok()
                .data("records", formattedRecords)
                .data("total", resultPage.getTotal());
    }


    @GetMapping("/{Id}")
    public Result getPost(@PathVariable int Id) {
        TORRENT torrent = postService.getById(Id);
        if (torrent == null) {
            return Result.error(404).setMessage("种子不存在");
        }

        // 获取所有评论（按创建时间排序）
        List<TorrentComment> allComments = commentService.list(
                new QueryWrapper<TorrentComment>()
                        .eq("post_id", Id)
                        .orderByAsc("create_time")
        );

        // 构建评论树形结构
        List<TorrentComment> rootComments = new ArrayList<>();
        Map<Integer, TorrentComment> commentMap = new HashMap<>();

        // 第一遍：初始化所有评论到map中
        for (TorrentComment comment : allComments) {
            comment.setReplies(new ArrayList<>()); // 初始化replies列表
            commentMap.put(comment.getId(), comment);
        }

        // 第二遍：构建父子关系
        for (TorrentComment comment : allComments) {
            if (comment.getParentId() == 0) {
                rootComments.add(comment);
            } else {
                TorrentComment parent = commentMap.get(comment.getParentId());
                if (parent != null) {
                    parent.getReplies().add(comment);
                }
            }
        }

        // 创建格式化后的torrent对象
        Map<String, Object> formattedTorrent = new HashMap<>();
        formattedTorrent.put("id", torrent.getId());
        formattedTorrent.put("torrentName", torrent.getTorrentName());
        formattedTorrent.put("description", torrent.getDescription());
        formattedTorrent.put("category", torrent.getCategory());
        formattedTorrent.put("region", torrent.getRegion());
        formattedTorrent.put("resolution", torrent.getResolution());
        formattedTorrent.put("subtitle", torrent.getSubtitle());
        formattedTorrent.put("size", formatFileSize(torrent.getSize())); // 格式化size
        formattedTorrent.put("hash", torrent.getHash());
        formattedTorrent.put("username", torrent.getUsername());
        formattedTorrent.put("filePath", torrent.getFilePath());
        formattedTorrent.put("like_count", torrent.getLike_count());
        formattedTorrent.put("reply_count", torrent.getReply_count());
        formattedTorrent.put("createTime", torrent.getCreateTime());

        return Result.ok()
                .data("torrent", formattedTorrent)
                .data("comments", rootComments);
    }

    // 点赞帖子
    @PostMapping("/{Id}/like")
    public Result likePost(
            @PathVariable int Id,
            @RequestParam("likerId") String likerId
    ) {
        postService.incrementLike(Id);
        // 通知发帖人
        TORRENT post = postService.getById(Id);
        if (post != null) {
            String authorId = post.getUsername(); // TORRENT 实体中发帖人字段是 username
            if (!authorId.equals(likerId)) {
                Notification n = new Notification();
                n.setUserId(authorId);  // 通知接收者 = 帖子作者
                n.setType("TORRENT_LIKE");
                n.setTitle("您的 Torrent 帖子被点赞");
                n.setContent("用户 "
                        + likerId
                        + " 点赞了您的 Torrent 帖子: \""
                        + post.getTorrentName()
                        + "\"");
                n.setTargetId(Id);
                n.setTargetType("TORRENT");
                n.setIsRead(false);
                n.setCreateTime(LocalDateTime.now());
                notificationService.saveNotification(n);
            }
        }

        return Result.ok();
    }

    @PostMapping("/{Id}/comments")
    public Result comment(
            @PathVariable int Id,
            @RequestBody TorrentComment comment
    ) {
        comment.setPostId(Id);
        comment.setCreateTime(LocalDateTime.now());
        comment.setLikeCount(0);
        if (comment.getParentId() == null) {
            comment.setParentId(0); // 默认一级评论
        }

        commentService.commentOnTorrentPost(comment); //通知作者
        TorrentComment newComment = commentService.getById(comment.getId());
        int updatedReplyCount = postService.getById(Id).getReply_count();

        return Result.ok()
                .data("comment", newComment)
                .data("newReplyCount", updatedReplyCount);
    }

    @PostConstruct //启动项目时候自动启动tracker服务器
    public String startTS() {
        try {
            trackerservice.startTracker();
            System.out.println("成功启动tracker服务器");
            return "成功启动tracker服务器";
        } catch (Exception e) {
            System.out.println("启动失败: " + e.getMessage());
            return "启动失败: " + e.getMessage();
        }
    }

    @PutMapping("/closeTS")
    public Result closeTS() {
        try {
            trackerservice.stopTracker();
            System.out.println("成功关闭tracker服务器");
            return Result.ok();
        } catch (Exception e) {
            System.out.println("启动失败: " + e.getMessage());
            return Result.error(500);
        }
    }


    //上传、下载、删除操作
    //post 要添加数据库 文件和tracker的信息
    //上传torrent
    @PostMapping(consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public Result addTorrent(HttpServletRequest request, @RequestPart("file") MultipartFile file, @RequestPart("body") TORRENT torrent) throws IOException {
        USER user = userController.getUserInRequest(request);

        // 1. 检查源文件
        if (!file.getOriginalFilename().endsWith(".torrent")) {
            return Result.error(403).data("message", "仅支持.torrent文件");
        }

        // 2. 确定存储目录（改为项目根目录下的torrents文件夹）
        String projectRoot = System.getProperty("user.dir");
        String uploadDir = projectRoot + File.separator + "torrents";
        File torrentsFolder = new File(uploadDir);
        if (!torrentsFolder.exists() && !torrentsFolder.mkdirs()) {
            return Result.error(404).data("message", "无法创建存储目录");
        }

        // 3. 获取文件名并创建上传文件路径
        String originalFilename = file.getOriginalFilename();
        File uploadedFile = new File(torrentsFolder, originalFilename);

        // 4. 处理文件保存
        try {
            // 如果文件已存在，给文件名加上一个时间戳，避免冲突
            if (uploadedFile.exists()) {
                String newFilename = System.currentTimeMillis() + "_" + originalFilename;
                uploadedFile = new File(torrentsFolder, newFilename);
            }

            // 复制文件
            Files.copy(file.getInputStream(), uploadedFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            System.out.println("文件保存成功: " + uploadedFile.getAbsolutePath());
        } catch (IOException e) {
            System.out.println("文件复制失败: " + e);
            return Result.error(404).data("message", "文件保存失败: " + e.getMessage());
        }

        // 5. 处理Tracker逻辑
        try {
            long fileSize = getTorrentSizeManual(uploadedFile);
            System.out.println("fileSize: " + fileSize);

            TrackedTorrent tt = trackerservice.addTrackedTorrent(uploadedFile.getAbsolutePath());

            // 6. 数据库操作
            torrent.setHash(tt.getHexInfoHash());
            torrent.setSize(fileSize);
            torrent.setUsername(user.getUsername());
            torrent.setFilePath(uploadedFile.getAbsolutePath());
            torrent.setCreateTime(LocalDateTime.now());
            torrentMapper.insertTorrent(torrent);
            System.out.println("Generated ID: " + torrent.getId());

            TrackedPeer tp = createTrackedPeer(request, tt, torrent.getId());
            tp.update(TrackedPeer.PeerState.COMPLETED, fileSize, 0, 0);
            tt.addPeer(tp);

            // 上传种子本人也上传一个下载记录，计算保种积分
            DOWNLOAD_TORRENT dt = new DOWNLOAD_TORRENT(user.getUsername(), torrent.getId(), torrent.getSize());
            downloadTorrentMapper.insertDownloadTorrent(dt);
            user.updateUpload(torrent.getSize());// 上传不计算折扣
            userMapper.updateUser(user);

            return Result.ok().data("message", "上传成功")
                    .data("path", uploadedFile.getAbsolutePath());

        } catch (Exception e) {
            System.out.println("Tracker或数据库操作失败: " + e);
            uploadedFile.delete(); // 回滚：删除已保存的文件
        }

        return Result.error(404).data("message", "上传失败");
    }

    //根据本机创建peer对象
    public TrackedPeer createTrackedPeer(HttpServletRequest request, TrackedTorrent trackedTorrent, int torrentid) throws IOException {
        // 修改为获取request请求的地址
        //String ip = request.getRemoteAddr();

        //获取本机地址
        InetAddress inetAddress = InetAddress.getLocalHost();
        String ip = inetAddress.getHostAddress();
        USER user = userController.getUserInRequest(request);

        int port = findAvailablePort(6881, 6999); // 使用固定端口

        // 创建Peer ID
        String peerIdString = UUID.randomUUID().toString();
        ByteBuffer peerId = ByteBuffer.wrap(peerIdString.getBytes());

        // 创建TrackedPeer实例
        TrackedPeer trackedPeer = new TrackedPeer(trackedTorrent, ip, port, peerId);

//        DOWNLOAD_TORRENT uwp = new DOWNLOAD_TORRENT(user.getUsername(), torrentid);


        return trackedPeer;
    }

    private int findAvailablePort(int start, int end) throws IOException {
        for (int port = start; port <= end; port++) {
            try (ServerSocket serverSocket = new ServerSocket(port)) {
                return port;
            } catch (IOException ignored) {
            }
        }
        throw new IOException("No available ports in range [" + start + ", " + end + "]");
    }

    //种子下载 根据id获取数据
    // 能成功获取到数据
    // 能够根据此值，完成对相关数据的修改
    @GetMapping("/downloadTorrent")
    public Result getTorrent(HttpServletRequest request,
                             @RequestParam int id,
                             @RequestParam String downloadPath) throws Exception {
        // 1. 获取种子和用户信息
        TORRENT torrent = torrentMapper.selectByID(id);
        String decodedPath = URLDecoder.decode(downloadPath, "UTF-8");
//        String normalizedPath = decodedPath.replace("\\", "/"); // 统一为斜杠
//        if (!normalizedPath.endsWith("/")) {
//            normalizedPath += "/"; // 确保以斜杠结尾
//        }
//
//        logger.info("下载请求 - 种子ID: {}, 原始路径: {}, 解码后路径: {}, 标准化路径: {}",
//                id, downloadPath, decodedPath, normalizedPath);
//
//        if (torrent == null) {
//            logger.error("尝试下载不存在的种子，ID: {}", id);
//            return Result.error(404).message("种子不存在");
//        }
        USER userDownload = userController.getUserInRequest(request);
        USER userUpload = userMapper.selectByUsername(torrent.getUsername());

        // 2. 记录下载日志
        logger.info("用户 {} 尝试下载种子 {} (上传者: {})",
                userDownload.getUsername(), id, userUpload.getUsername());

        // 3. 检查分享率（如果是自己的种子则跳过检查）
        if (!userDownload.getUsername().equals(userUpload.getUsername()) &&
                !userDownload.allowDownload(torrent.getSize())) {
            logger.warn("用户 {} 分享率不足，无法下载种子 {}",
                    userDownload.getUsername(), id);
            return Result.error(409).message("分享率不足下载此资源");
        }

        // 4. 执行下载
//        try {
            // 4.1 调用客户端服务下载

            clientservice.downloadTorrent(torrent.getFilePath(), decodedPath, userDownload.getUsername());
            // 4.2 更新Tracker信息
            TrackedTorrent tt = trackerservice.getTrackedTorrent(torrent.getHash());
            TrackedPeer tp = createTrackedPeer(request, tt, torrent.getId());
            tp.update(TrackedPeer.PeerState.COMPLETED, 0, 0, torrent.getSize());
            tt.addPeer(tp);
            trackerservice.serTracker();

//        // 通知下载完成
//        Notification downloadCompleteNotice = new Notification();
//        downloadCompleteNotice.setUserId(userDownload.getUsername());
//        downloadCompleteNotice.setType("DOWNLOAD_COMPLETE");
//        downloadCompleteNotice.setTitle("下载完成提醒");
//        downloadCompleteNotice.setContent(
//                "您下载的种子 \"" + torrent.getTorrentName() + "\" 已经成功完成。"
//        );
//        downloadCompleteNotice.setTargetId(id);
//        downloadCompleteNotice.setTargetType("TORRENT");
//        downloadCompleteNotice.setIsRead(false);
//        downloadCompleteNotice.setCreateTime(LocalDateTime.now());
//        notificationService.saveNotification(downloadCompleteNotice);

            // 4.3 更新用户数据（只有下载他人种子时才更新）
            if (!userDownload.getUsername().equals(userUpload.getUsername())) {
                userDownload.updateDownload(SizeCalculation.getDownload(torrent.getSize(), discountMapper, torrent.getCreateTime()));
                userUpload.updateUpload(SizeCalculation.getUpload(torrent.getSize(), discountMapper, torrent.getCreateTime()));
                userMapper.updateUser(userUpload);
                userMapper.updateUser(userDownload);

                DOWNLOAD_TORRENT dt = new DOWNLOAD_TORRENT(userDownload.getUsername(), torrent.getId(), torrent.getSize());
                downloadTorrentMapper.insertDownloadTorrent(dt);
            }

            logger.info("用户 {} 成功下载种子 {}", userDownload.getUsername(), id);
            return Result.ok().data("fileName", torrent.getTorrentName())
                    .data("taskId", torrent.getHash());
//        }catch (Exception e) {
//            logger.error("下载种子失败", e);
//            return Result.error(500).message("下载失败: " + e.getMessage());
//        }
    }

    @GetMapping("/getProgress")
    public Result getProgress(HttpServletRequest request) throws Exception {
        USER user = userController.getUserInRequest(request);
        String username = user.getUsername();
        List<String[]> tasks = clientservice.getTasks();
        Map<String, Double> progresses = new HashMap<>();

        for (String[] task : tasks) {
            if (task[0].equals(username)) {
                try {
                    double progress = clientservice.getDownloadProgress(task[1]);
                    // 确保进度在0-1之间
                    progress = Math.max(0, Math.min(1, progress));
                    progresses.put(task[1], progress);
                } catch (Exception e) {
                    // 如果出现异常，假设下载已完成
                    progresses.put(task[1], 1.0);
                }
            }
        }
        return Result.ok().data("progresses", progresses);
    }

    //根据id删除
    @DeleteMapping("/deleteTorrent/{id}")
    public Result deleteTorrent(HttpServletRequest request, @PathVariable("id") int id) throws Exception {
        // 1. 验证用户权限
        USER currentUser = userController.getUserInRequest(request);
        TORRENT torrent = torrentMapper.selectByID(id);

        if (torrent == null) {
            return Result.error(404).setMessage("种子不存在");
        }

        // 3. 删除关联记录（分步骤）
        try {
            // 3.1 先删除下载记录
            downloadTorrentMapper.deleteByTorrentId(id);

            // 3.2 删除种子文件
            if (Files.exists(Paths.get(torrent.getFilePath()))) {
                Files.delete(Paths.get(torrent.getFilePath()));
            }

            // 3.3 最后删除种子记录
            int deleteCount = torrentMapper.deleteTorrent(id);
            if (deleteCount == 0) {
                return Result.error(404).setMessage("种子删除失败，可能已被删除");
            }

            return Result.ok().message("种子删除成功");

        } catch (IOException e) {
            logger.error("文件删除失败", e);
            return Result.error(500).setMessage("文件删除失败: " + e.getMessage());
        } catch (Exception e) {
            logger.error("种子删除失败", e);
            return Result.error(500).setMessage("种子删除失败: " + e.getMessage());
        }
    }



    @GetMapping("/search")
    public Result searchTorrents(
            @RequestParam String keyword,
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "5") int size) {

        // 1. 把搜索词拆分成单词（中英文混合拆分）
        List<String> keywords = splitKeywords(keyword);

        // 2. 构建动态查询条件（只搜索种子名称）
        QueryWrapper<TORRENT> queryWrapper = new QueryWrapper<>();
        for (String word : keywords) {
            queryWrapper.and(wrapper ->
                    // 使用LOWER函数将数据库字段和搜索词都转为小写
                    wrapper.apply("LOWER(torrent_name) LIKE LOWER({0})", "%" + word + "%")
            );
        }
        queryWrapper.orderByDesc("create_time");

        // 3. 执行分页查询
        IPage<TORRENT> ipage = new Page<>(page, size);
        IPage<TORRENT> resultPage = torrentMapper.selectPage(ipage, queryWrapper);

        return Result.ok()
                .data("records", resultPage.getRecords())
                .data("total", resultPage.getTotal());
    }

    /**
     * 拆分关键词（支持中英文混合）
     */
    private List<String> splitKeywords(String keyword) {
        List<String> keywords = new ArrayList<>();
        Pattern pattern = Pattern.compile("([\\u4e00-\\u9fa5])|([a-zA-Z0-9]+)");
        Matcher matcher = pattern.matcher(keyword);
        while (matcher.find()) {
            keywords.add(matcher.group());
        }
        return keywords;
    }


    @GetMapping("/get/torrentMyself")
    public Result getTorrentByKey(
            HttpServletRequest request,
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "5") int size) throws Exception {

        USER user = userController.getUserInRequest(request);
        int offset = (page - 1) * size;
        List<TORRENT> pagedList = torrentMapper.selectTorrentByUsernameWithPage(user.getUsername(), offset, size);

        // 为每个种子添加下载次数和格式化后的文件大小
        List<Map<String, Object>> resultList = new ArrayList<>();
        for (TORRENT torrent : pagedList) {
            Map<String, Object> torrentMap = new HashMap<>();
            torrentMap.put("torrent", torrent);
            torrentMap.put("downloadCount", getDownloadCount(torrent.getId()));
            torrentMap.put("formattedSize", formatFileSize(torrent.getSize())); // 添加格式化后的文件大小
            resultList.add(torrentMap);
        }

        int total = torrentMapper.countByUsername(user.getUsername());

        Page<Map<String, Object>> pageResult = new Page<>(page, size);
        pageResult.setRecords(resultList);
        pageResult.setTotal(total);

        return Result.ok()
                .data("records", pageResult.getRecords())
                .data("total", pageResult.getTotal());
    }


    // 测试类
    @GetMapping("/test")
    public Result updatePeer(HttpServletRequest request) throws IOException {
        TORRENT torrent = torrentMapper.selectByID(90);
        USER user = userController.getUserInRequest(request);

        DOWNLOAD_TORRENT dt = new DOWNLOAD_TORRENT(user.getUsername(), torrent.getId(), torrent.getSize());
        downloadTorrentMapper.insertDownloadTorrent(dt);
        user.updateUpload(torrent.getSize());// 上传不计算折扣
        userMapper.updateUser(user);
        return Result.ok();
    }

    //获取torrent文件大小
    public static long getTorrentSizeManual(File torrentFile) throws IOException {
        byte[] data = Files.readAllBytes(torrentFile.toPath());
        ByteArrayInputStream bis = new ByteArrayInputStream(data);
        BEValue decoded = BDecoder.bdecode(bis);
        Map<String, BEValue> torrentData = decoded.getMap();
        Map<String, BEValue> info = torrentData.get("info").getMap();

        // 单文件情况
        if (info.containsKey("length")) {
            return info.get("length").getLong();
        }
        // 多文件情况
        else if (info.containsKey("files")) {
            long totalSize = 0;
            for (BEValue file : info.get("files").getList()) {
                totalSize += file.getMap().get("length").getLong();
            }
            return totalSize;
        }
        throw new IOException("Invalid torrent file");
    }

    private int getDownloadCount(int torrentId) {
        QueryWrapper<DOWNLOAD_TORRENT> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("torrentid", torrentId);
        return downloadTorrentMapper.selectCount(queryWrapper).intValue();
    }

    private String formatFileSize(long size) {
        if (size < 1024) {
            return size + " B";
        }
        double kb = size / 1024.0;
        if (kb < 1024) {
            return String.format("%.2f KB", kb);
        }
        double mb = kb / 1024.0;
        if (mb < 1024) {
            return String.format("%.2f MB", mb);
        }
        double gb = mb / 1024.0;
        return String.format("%.2f GB", gb);
    }

}