//controller 文件夹中文件类似于view
// usermapper 实现用户登录注册等内容

package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.*;
import com.ptp.ptplatform.mapper.DownloadTorrentMapper;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.mapper.InviteCodeMapper;
import com.ptp.ptplatform.service.UserLevelService;
import com.ptp.ptplatform.utils.SizeCalculation;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import com.ptp.ptplatform.utils.Result;
import com.ptp.ptplatform.utils.JwtUtils;
import com.ptp.ptplatform.entity.USER;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;

@RestController
@RequestMapping("/user")
@CrossOrigin //启用跨域
public class UserController {

    @Autowired
    private UserMapper userMapper;
    @Autowired
    private InviteCodeMapper inviteCodeMapper;
    @Autowired
    private UserLevelService userLevelService;
    @Autowired
    private DownloadTorrentMapper downloadTorrentMapper;

    @GetMapping("/userInfo")
    public Result userInfo(HttpServletRequest request) {
        try {
            USER user = this.getUserInRequest(request);
            if (user == null) {
                return Result.error(404).setMessage("用户未找到");
            }

            // 创建一个Map来组织返回数据
            Map<String, Object> userInfo = new HashMap<>();
            userInfo.put("username", user.getUsername());
            userInfo.put("authority", user.getAuthority().name());
            userInfo.put("level", user.getLevel());
            userInfo.put("registTime", user.getRegistTime());
            userInfo.put("upload", user.getUpload());
            userInfo.put("download", user.getDownload());
            userInfo.put("shareRate", user.getShareRate());
            userInfo.put("magicPoints", user.getMagicPoints());
            userInfo.put("lastLogin", user.getLastLogin());

            return Result.ok().data(userInfo);
        } catch (Exception e) {
            return Result.error(500).setMessage("获取用户信息失败");
        }
    }


    @PostMapping("/login") //用户登录
    public Result login(String username, String password) {
        USER user = userMapper.selectByUsername(username);

        // 检查用户是否存在
        if (user == null) {
            return Result.error(404).setMessage("用户不存在");
        }
        if(user.getAuthority() == USER.Authority.BAN) {
            return Result.error(403).setMessage("用户被封禁，禁止登录");
        }

        System.out.println("password" + user.getPassword());

        // 比较用户输入密码值是否正确
        if (user.getPassword().equals(password)) {
            String token = JwtUtils.generateToken(user.getUsername());

            user.setLastLogin(new Date());
            userMapper.updateUser(user);
            return Result.ok().data("token", token);  // 返回令牌给前端
        } else {
            return Result.error(404).setMessage("密码错误");
        }
    }

    @PostMapping("/regist")
    public Result regist(String username, String password, String code) {
        USER userCheck = userMapper.selectByUsername(username);
        if (userCheck == null) {
            //获取邀请码
            INVITE_CODE inviteCode = inviteCodeMapper.selectByCode(code);
            if (inviteCode != null) {
                System.out.println(inviteCode.getIsUsed());

                if (!inviteCode.getIsUsed()) {
                    Date time = new Date();
                    USER user = new USER(username, password, time);

                    userMapper.insertUser(user);
                    inviteCodeMapper.updateCodeUser(code);

                    return Result.ok().setMessage("新建用户成功");
                } else {
                    return Result.error(404).setMessage("邀请码已经被使用，注册失败");
                }

            } else {
                return Result.error(404).setMessage("邀请码不存在，注册失败");
            }

        } else {
            return Result.error(404).setMessage("用户名已存在，注册失败");
        }


    }

    @GetMapping("/allowDownload")
    public Result allowDownload(HttpServletRequest request) {
        USER user = this.getUserInRequest(request);
        // 直接返回字节值
        Map<String, Object> dataMap = new HashMap<>();
        long remaining = user.getUpload() * 2 - user.getDownload(); // 剩余额度=上传量×2-下载量
        long used = user.getDownload(); // 已使用=下载量
        long total = used + remaining; // 总额度=已使用+剩余

        dataMap.put("total", total);
        dataMap.put("used", used);
        dataMap.put("remaining", remaining);
        return Result.ok().data(dataMap);
    }


    //修改用户密码
    @PutMapping("/password")
    public Result updatePassword(HttpServletRequest request, @RequestBody Map<String, String> passwordMap) {
        USER user = this.getUserInRequest(request);

        String oldPassword = passwordMap.get("oldPassword");
        String newPassword = passwordMap.get("newPassword");

        if (user.getPassword().equals(oldPassword)) {
            user.setPassword(newPassword);
            userMapper.updateUser(user);

            return Result.ok().setMessage("修改密码成功");
        }
        return Result.error(404).setMessage("原密码不正确");
    }

    //搜索用户
    @GetMapping("/searchUser")
    public Result searchUser(@RequestParam(value = "key", required = false) String key, HttpServletRequest request) {
        // 从请求中获取当前用户
        USER user = this.getUserInRequest(request);
        String username = user.getUsername();

        // 检查是否提供了搜索关键词
        if (key != null && !key.isEmpty()) {
            List<USER> searchResults = userMapper.searchUsername(username, key);
            return Result.ok().data("data", searchResults);
        } else {
            // 如果没有提供关键词，返回默认结果或者一个错误消息
            return Result.error(404).setMessage("请提供搜索关键词");
        }
    }

    //获取到全部用户
    @GetMapping("/allUser")
    public Result getAllUser(){
        List<USER> users = userMapper.selectAllUsers();

        return Result.ok().data("data", users);
    }


    //修改用户权限
    // 传入数据 username 要修改到的权限
    @PutMapping("/changeAuthority")
    public Result changeAuthority(HttpServletRequest request, @RequestBody Map<String, String> authorityMap) {
        String authority = authorityMap.get("authority");
        String changeUsername = authorityMap.get("changeUsername");

        USER changeUser = userMapper.selectByUsername(changeUsername);
        changeUser.setAuthority(USER.Authority.valueOf(authority));

        if(userMapper.updateUser(changeUser) > 0){
            return Result.ok().setMessage("修改用户权限成功");
        }
        return Result.error(400).setMessage("修改用户权限失败");
    }

    public USER getUserInRequest(HttpServletRequest request) {
        String authHeader = request.getHeader("Authorization");
        System.out.println("authHeader" + authHeader);
        if (authHeader == null) {
            return null;
        }
        try {
            String username = JwtUtils.getClaimByToken(authHeader).getSubject();
            System.out.println("username" + username);
            return userMapper.selectByUsername(username);
        } catch (Exception e) {
            e.printStackTrace();  // 添加日志
            return null;
        }
    }

    // 用户等级更新
    @PostMapping("/refreshLevel")
    public Result refresh() {
        userLevelService.refreshAllUserLevels();
        return Result.ok();
    }

    //计算保种积分
    @GetMapping("/magicPoint")
    public Result getMagicPoint(HttpServletRequest request) {
        USER user = this.getUserInRequest(request);
        List<DOWNLOAD_TORRENT> dts = downloadTorrentMapper.selectByUsername(user.getUsername());

        for (DOWNLOAD_TORRENT dt : dts) {
            long during = ChronoUnit.DAYS.between(dt.getUpdated(), LocalDateTime.now());
            double size = SizeCalculation.byteToGB(dt.getFileSize());
            dt.setUpdated(LocalDateTime.now());
            user.addMagicPoints((long) (during * size));

            downloadTorrentMapper.updateDT(dt);
        }

        userMapper.updateUser(user);
        return Result.ok().data("data", user);
    }

    //保重积分兑换下载量
    //1gb资源代表一保重积分
    @PostMapping("/exchangeUpload")
    public Result exchangeUpload(
            HttpServletRequest request,
            @RequestBody Map<String, Long> requestBody // 接收 JSON 对象
    ) {
        USER user = this.getUserInRequest(request);
        Long magicPoint = requestBody.get("magicPoint"); // 获取字段值

        if (magicPoint == null) {
            return Result.error(400).message("magicPoint 不能为空");
        }

        System.out.println("magicPoint" + magicPoint);
        System.out.println("user magicPoint" + user.getMagicPoints());

        if (user.getMagicPoints() >= magicPoint) {
            user.reduceMagicPoints(magicPoint);
            user.updateUpload(SizeCalculation.GBToByte(magicPoint));
            userMapper.updateUser(user);
            return Result.ok().message("兑换上传量成功");
        }
        return Result.error(409).message("兑换值超出最大魔力值，兑换失败");
    }
}
