package com.ptp.ptplatform.service;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;

import com.ptp.ptplatform.entity.Notification;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ptp.ptplatform.entity.USER;
import com.ptp.ptplatform.mapper.UserMapper;

import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class UserLevelService {

    @Autowired
    private UserMapper userMapper;
    @Autowired
    private NotificationService notificationService; // 新注入，用于发送通知
    private static final double BYTES_PER_GB = 1024d * 1024 * 1024;

    // 每周一凌晨 2 点触发
    @Scheduled(cron = "0 0 2 * * MON")
    @Transactional
    public void refreshAllUserLevels() {
        log.info("===== 开始刷新所有用户等级 =====");
        LocalDate today = LocalDate.now();

        // 1. 查询数据库里所有用户
        List<USER> users = userMapper.selectList(null);
        log.info(">> 从数据库查到 {} 个用户", users.size());

        for (USER u : users) {
            double shareRate    = u.getShareRate();                // 当前分享率
            double uploadGb     = u.getUpload() / BYTES_PER_GB;    // 上传大小 (GB)
            long   magicPoints  = u.getMagicPoints();              // 魔力值
            Date   regTime      = u.getRegistTime();               // 注册时间
            LocalDate regDate   = regTime.toInstant()
                    .atZone(ZoneId.systemDefault())
                    .toLocalDate();
            long   monthsSince  = ChronoUnit.MONTHS.between(regDate, today);

            log.info("[LevelCheck] 用户={} shareRate={} uploadGB={} magicPoints={} months={}",
                    u.getUsername(), shareRate, uploadGb, magicPoints, monthsSince);

            // 2. 先记录旧等级
            int oldLevel = u.getLevel();

            // 3. 根据业务规则计算新等级
            int newLevel = calcLevel(u, uploadGb, shareRate, magicPoints, monthsSince);

            if (newLevel != oldLevel) {
                log.info("[LevelUpdate] 用户={} 等级: {} -> {}", u.getUsername(), oldLevel, newLevel);

                // 4. 更新用户实体的等级并写库
                u.setLevel(newLevel);
//                userMapper.updateById(u);
                userMapper.updateUser(u);

                // 5. 构造并保存“等级提升”通知
                Notification n = new Notification();
                n.setUserId(u.getUsername());                    // 通知接收者 = 用户自己
                n.setType("LEVEL_UP");                            // 通知类型：LEVEL_UP
                n.setTitle("恭喜您，用户等级已提升");              // 通知标题
                n.setContent("您的用户等级已从 "
                        + oldLevel
                        + " 级提升至 "
                        + newLevel
                        + " 级！");                         // 通知内容
                n.setTargetId(null);                              // 等级通知无需跳转
                n.setTargetType(null);
                n.setIsRead(false);
                n.setCreateTime(java.time.LocalDateTime.now());

                notificationService.saveNotification(n);
            }
        }

        log.info("===== 刷新完毕 =====");
    }


    /**
     * 根据各项指标和注册时长计算用户等级
     */
    private int calcLevel(
            USER u,
            double uploadGb,
            double shareRate,
            long magicPoints,
            long monthsSince
    ) {
        if (shareRate   >= 1.2    &&
                magicPoints >= 100_000 &&
                uploadGb    >= 10_240  &&
                monthsSince >= 12) {
            return 5;
        }
        if (shareRate   >= 1.0    &&
                magicPoints >=  50_000 &&
                uploadGb    >=  5_120  &&
                monthsSince >= 6) {
            return 4;
        }
        if (shareRate   >= 0.8    &&
                magicPoints >=  20_000 &&
                uploadGb    >=  1_024  &&
                monthsSince >= 3) {
            return 3;
        }
        if (shareRate   >= 0.6    &&
                magicPoints >=   5_000 &&
                uploadGb    >=    200  &&
                monthsSince >= 1) {
            return 2;
        }
        if (shareRate   >= 0.4    &&
                magicPoints >=   1_000 &&
                uploadGb    >=     50) {
            return 1;
        }
        return 0;
    }
}
