package com.ptp.ptplatform.service.impl;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.ptp.ptplatform.entity.DOWNLOAD_TORRENT;
import com.ptp.ptplatform.entity.RECOMMEND_HISTORY;
import com.ptp.ptplatform.entity.TORRENT;
import com.ptp.ptplatform.mapper.DownloadTorrentMapper;
import com.ptp.ptplatform.mapper.RecommendHistoryMapper;
import com.ptp.ptplatform.mapper.TorrentMapper;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.service.RecommenderService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

// src/main/java/com/yourpackage/service/impl/RecommenderServiceImpl.java
@Service
public class RecommenderServiceImpl implements RecommenderService {

    private final TorrentMapper torrentRepository;
    private final DownloadTorrentMapper downloadRepository;
    private final UserMapper userRepository;
    private final RecommendHistoryMapper historyRepository;

    @Autowired
    public RecommenderServiceImpl(TorrentMapper torrentRepository,
                                  DownloadTorrentMapper downloadRepository,
                                  UserMapper userRepository,
                                  RecommendHistoryMapper historyRepository) {
        this.torrentRepository = torrentRepository;
        this.downloadRepository = downloadRepository;
        this.userRepository = userRepository;
        this.historyRepository = historyRepository;
    }

    @Override
    public List<TORRENT> recommendForUser(String username, int limit) {
        // 1. 获取用户已经推荐过的资源ID
        Set<Integer> alreadyRecommended = historyRepository.findRecommendedTorrentIdsByUser(username);

        // 2. 获取用户已经下载过的资源ID
        Set<Integer> alreadyDownloaded = getDownloadedTorrentIds(username);

        // 3. 合并排除集合
        Set<Integer> excludedIds = new HashSet<>();
        excludedIds.addAll(alreadyRecommended);
        excludedIds.addAll(alreadyDownloaded);
        System.out.println("excludedIds: " + excludedIds);

        // 4. 获取原始推荐结果（80%）
        List<TORRENT> originalRecommendations = getOriginalRecommendations(username,
                (int)(limit * 0.8), excludedIds);
        System.out.println("originalRecommendations 数量: " + originalRecommendations.size());

        // 5. 获取随机推荐（20%），同样排除已推荐和已下载的
        List<TORRENT> randomRecommendations = getRandomRecommendations(
                limit - originalRecommendations.size(), excludedIds);
        System.out.println("randomRecommendations 数量: " + randomRecommendations.size());

        // 6. 合并结果
        List<TORRENT> finalRecommendations = new ArrayList<>();
        finalRecommendations.addAll(originalRecommendations);
        finalRecommendations.addAll(randomRecommendations);
        Collections.shuffle(finalRecommendations);
        System.out.println("最终推荐数量: " + finalRecommendations.size());

        // 7. 记录本次推荐
        recordRecommendations(username, finalRecommendations);

        return finalRecommendations.stream().limit(limit).collect(Collectors.toList());
    }

    private Set<Integer> getDownloadedTorrentIds(String username) {
        return downloadRepository.findByUsername(username).stream()
                .map(DOWNLOAD_TORRENT::getTorrentid)
                .collect(Collectors.toSet());
    }

    private List<TORRENT> getOriginalRecommendations(String username, int limit, Set<Integer> excludedIds) {
        // 1. 获取用户下载历史
        List<DOWNLOAD_TORRENT> userDownloads = downloadRepository.findByUsername(username);
        if (userDownloads.isEmpty()) {
            return getPopularTorrents(limit, excludedIds);
        }

        // 2. 收集相似用户
        Set<String> similarUsers = new HashSet<>();
        for (DOWNLOAD_TORRENT download : userDownloads) {
            List<String> users = downloadRepository.findUsersByTorrentId(download.getTorrentid());
            similarUsers.addAll(users);
        }
        similarUsers.remove(username);

        // 3. 获取相似用户喜欢的种子（排除已推荐和已下载的）
        Map<Integer, Integer> torrentScores = new HashMap<>();
        for (String similarUser : similarUsers) {
            downloadRepository.findByUsername(similarUser).stream()
                    .filter(d -> !excludedIds.contains(d.getTorrentid()))
                    .forEach(d -> torrentScores.merge(d.getTorrentid(), 1, Integer::sum));
        }

        // 4. 排序并获取推荐
        return torrentScores.entrySet().stream()
                .sorted(Map.Entry.<Integer, Integer>comparingByValue().reversed())
                .limit(limit)
                .map(e -> torrentRepository.findById(e.getKey()))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private List<TORRENT> getRandomRecommendations(int count, Set<Integer> excludedIds) {
        QueryWrapper<TORRENT> query = new QueryWrapper<>();
        if (!excludedIds.isEmpty()) {
            query.notIn("id", excludedIds);
        }
        return torrentRepository.selectList(query
                .orderByAsc("RAND()")
                .last("LIMIT " + count));
    }

    private List<TORRENT> getPopularTorrents(int limit, Set<Integer> excludedIds) {
        QueryWrapper<TORRENT> query = new QueryWrapper<>();
        if (!excludedIds.isEmpty()) {
            query.notIn("id", excludedIds);
        }
        return torrentRepository.selectList(query
                .orderByDesc("like_count")
                .last("LIMIT " + limit));
    }

    private void recordRecommendations(String username, List<TORRENT> recommendations) {
        List<RECOMMEND_HISTORY> newRecords = recommendations.stream()
                .filter(t -> historyRepository.existsByUserAndTorrent(username, t.getId()) == 0)
                .map(t -> new RECOMMEND_HISTORY(username, t.getId()))
                .collect(Collectors.toList());

        if (!newRecords.isEmpty()) {
            newRecords.forEach(historyRepository::insert);
        }
    }

    @Override
    public List<TORRENT> getSimilarTorrents(Integer torrentId, int limit) {
        TORRENT target = torrentRepository.findById(torrentId);
        if (target == null) {
            return Collections.emptyList();
        }

        return torrentRepository.selectList(new QueryWrapper<TORRENT>()
                .eq("category", target.getCategory())
                .ne("id", torrentId)
                .orderByDesc("like_count")
                .last("LIMIT " + limit));
    }

    @Override
    public List<TORRENT> getPopularTorrents(int limit) {
        return torrentRepository.selectList(new QueryWrapper<TORRENT>()
                .orderByDesc("like_count")
                .last("LIMIT " + limit));
    }

    @Override
    public void markRecommendationAsShown(String username, Integer torrentId) {
        historyRepository.markAsShown(username, torrentId);
    }

    @Override
    public void markRecommendationAsClicked(String username, Integer torrentId) {
        historyRepository.markAsClicked(username, torrentId);
    }
}