package com.ptp.ptplatform.service.impl;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ptp.ptplatform.entity.RequestComment;
import com.ptp.ptplatform.entity.RequestPost;
import com.ptp.ptplatform.entity.Notification;
import com.ptp.ptplatform.mapper.RequestCommentMapper;
import com.ptp.ptplatform.mapper.RequestPostMapper;
import com.ptp.ptplatform.service.RequestPostService;
import com.ptp.ptplatform.service.NotificationService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;

@Service
public class RequestPostServiceImpl extends ServiceImpl<RequestPostMapper, RequestPost> implements RequestPostService {
    @Autowired
    private RequestPostMapper requestPostMapper;
    @Autowired
    private RequestCommentMapper requestCommentMapper;
    @Autowired
    private NotificationService notificationService;
    @Override
    @Transactional
    public void likePost(Integer postId, String likerId) {
        // 1. 查询帖子本身，获取 authorId 和 title
        RequestPost post = requestPostMapper.selectById(postId);
        if (post == null) {
            throw new RuntimeException("帖子不存在，ID=" + postId);
        }

        // 2. 更新 like_count 字段
        requestPostMapper.update(
                null,
                new UpdateWrapper<RequestPost>()
                        .eq("id", postId)
                        .setSql("like_count = like_count + 1")
        );

        // 3. 给帖子作者发通知（排除自己给自己点赞的情况）
        String authorId = post.getAuthorId();
        if (!authorId.equals(likerId)) {
            Notification n = new Notification();
            n.setUserId(authorId);               // 通知接收方 = 帖子作者
            n.setType("POST_LIKE");              // 通知类型，可前端约定
            n.setTitle("您的帖子被点赞");        // 通知标题
            n.setContent("用户 "
                    + likerId
                    + " 点赞了您的帖子: \""
                    + post.getTitle()
                    + "\"");
            n.setTargetId(postId);               // 通知跳转时可指向该帖子
            n.setTargetType("HelpPost");         // 前端可根据此值决定跳转逻辑
            notificationService.saveNotification(n);
        }
    }
    @Override
    @Transactional
    public void incrementLike(Integer postId) {
        this.update(
                null,
                new UpdateWrapper<RequestPost>()
                        .eq("id", postId)
                        .setSql("like_count = like_count + 1")
        );
    }

    @Override
    @Transactional
    public void incrementReplyCount(Integer postId) {
        this.update(
                null,
                new UpdateWrapper<RequestPost>()
                        .eq("id", postId)
                        .setSql("reply_count = reply_count + 1")
        );
    }

    @Override
    @Transactional
    public void commentOnRequestPost(RequestComment comment) {
        // 1. 将新的 HelpComment 插入数据库
        //    如果 parentId != 0，则说明这是一条对子评论的回复；如果 parentId = 0，则这就是对帖子的一级评论。
        requestCommentMapper.insert(comment);

        // 2. 帖子回复数 +1（reply_count 字段）
        Integer postId = comment.getPostId();
        requestPostMapper.update(
                null,
                new UpdateWrapper<RequestPost>()
                        .eq("id", postId)
                        .setSql("reply_count = reply_count + 1")
        );

        // 3. 给帖子作者发一条通知（如果评论人不是作者自己）
        RequestPost post = requestPostMapper.selectById(postId);
        if (post != null) {
            String authorId = post.getAuthorId();            // 帖子作者的 ID
            String commenterId = comment.getAuthorId();      // 当前发表评论的用户 ID
            if (!authorId.equals(commenterId)) {
                Notification n = new Notification();
                n.setUserId(authorId);                      // 通知接收人 = 帖子作者
                n.setType("POST_REPLY");                    // 通知类型，可与前端约定
                n.setTitle("您的帖子有了新回复");             // 通知标题
                // 通知内容示例： 用户 <commenterId> 评论了您的帖子: "评论内容"
                n.setContent("用户 "
                        + commenterId
                        + " 评论了您的帖子: \""
                        + comment.getContent()
                        + "\"");
                n.setTargetId(postId);                      // targetId 可指向该帖子
                n.setTargetType("HelpPost");                // targetType = "HelpPost"
                n.setIsRead(false);
                n.setCreateTime(LocalDateTime.now());
                notificationService.saveNotification(n);
            }
        }
    }
}