package com.ptp.ptplatform.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.ptp.ptplatform.entity.Notification;
import com.ptp.ptplatform.entity.TORRENT;
import com.ptp.ptplatform.entity.TorrentComment;
import com.ptp.ptplatform.mapper.TorrentCommentMapper;
import com.ptp.ptplatform.mapper.TorrentMapper;
import com.ptp.ptplatform.service.NotificationService;
import com.ptp.ptplatform.service.TorrentCommentService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class TorrentCommentServiceImpl extends ServiceImpl<TorrentCommentMapper, TorrentComment> implements TorrentCommentService {
    @Autowired
    private TorrentCommentMapper torrentCommentMapper;

    @Autowired
    private TorrentMapper torrentMapper;

    @Autowired
    private NotificationService notificationService;

    @Override
    @Transactional
    public void incrementLike(int commentId) {
        this.update(null,
                new com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper<TorrentComment>()
                        .eq("id", commentId)
                        .setSql("like_count = like_count + 1")
        );
    }

    @Override
    @Transactional(readOnly = true)
    public List<TorrentComment> getReplies(int parentId) {
        return this.baseMapper.selectList(
                new QueryWrapper<TorrentComment>()
                        .eq("parent_id", parentId)
                        .orderByAsc("create_time")
        );
    }

    @Override
    @Transactional
    public void commentOnTorrentPost(TorrentComment comment) {
        // ————————————————
        // 1. 插入新评论到 torrent_comments 表
        //    插入后，MyBatis-Plus 会自动将自增主键 id 回写到 comment.getId()
        // ————————————————
        comment.setCreateTime(LocalDateTime.now());
        comment.setLikeCount(0);
        torrentCommentMapper.insert(comment);

        // ————————————————
        // 2. 更新对应 TORRENT 帖子的 reply_count 字段 +1
        // ————————————————
        Integer postId = comment.getPostId();
        torrentMapper.update(
                null,
                new UpdateWrapper<TORRENT>()
                        .eq("id", postId)
                        .setSql("reply_count = reply_count + 1")
        );

        // ————————————————
        // 3. 给该帖子的作者发通知（如果评论人 != 作者）
        // ————————————————
        TORRENT post = torrentMapper.selectById(postId);
        if (post != null) {
            String postAuthorId = post.getUsername();
            String commenterId  = comment.getAuthorId();

            if (!postAuthorId.equals(commenterId)) {
                Notification n = new Notification();
                n.setUserId(postAuthorId);                    // 通知接收人 = 帖子作者
                n.setType("TORRENT_POST_REPLY");               // 通知类型，可自定义
                n.setTitle("您的 Torrent 帖子有新回复");       // 通知标题
                n.setContent(
                        "用户 " + commenterId + " 评论了您的 Torrent 帖子: \"" +
                                comment.getContent() + "\""
                );
                n.setTargetId(postId);                         // 目标 ID = 帖子 ID
                n.setTargetType("TORRENT");                    // 目标类型 = TORRENT
                n.setIsRead(false);                            // 默认未读
                n.setCreateTime(LocalDateTime.now());
                notificationService.saveNotification(n);
            }

            if (comment.getParentId() != null && comment.getParentId() > 0) {
                Integer parentCommentId = comment.getParentId();
                TorrentComment parentComment = torrentCommentMapper.selectById(parentCommentId);
                if (parentComment != null) {
                    String parentAuthorId = parentComment.getAuthorId();
                    // 排除顶层评论 + 排除作者自己给自己回复
                    if (!parentAuthorId.equals(commenterId)) {
                        Notification n2 = new Notification();
                        n2.setUserId(parentAuthorId);
                        n2.setType("TORRENT_COMMENT_REPLY");
                        n2.setTitle("您的评论有新回复");
                        n2.setContent(
                                "用户 " + commenterId + " 回复了您的评论: \"" +
                                        comment.getContent() + "\""
                        );
                        n2.setTargetId(parentCommentId);
                        n2.setTargetType("TorrentComment");
                        n2.setIsRead(false);
                        n2.setCreateTime(LocalDateTime.now());
                        notificationService.saveNotification(n2);
                    }
                }
            }
        }
    }
}