package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.ANNOUNCEMENT;
import com.ptp.ptplatform.mapper.AnnouncementMapper;
import com.ptp.ptplatform.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

class AnnouncementControllerTest {

    @Mock
    private AnnouncementMapper announcementMapper;

    @InjectMocks
    private AnnouncementController announcementController;

    private ANNOUNCEMENT mockAnnouncement;
    private final LocalDateTime now = LocalDateTime.now();

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);

        // 初始化测试数据
        mockAnnouncement = new ANNOUNCEMENT();
        mockAnnouncement.setId(1);
        mockAnnouncement.setTitle("测试公告");
        mockAnnouncement.setContent("这是测试公告内容");
        mockAnnouncement.setCreateTime(now);
    }

    @Test
    void listAll_ShouldReturnAllAnnouncements() {
        // 准备测试数据
        ANNOUNCEMENT a2 = new ANNOUNCEMENT();
        a2.setId(2);
        a2.setTitle("另一个公告");

        List<ANNOUNCEMENT> announcements = Arrays.asList(mockAnnouncement, a2);

        // 模拟Mapper行为
        when(announcementMapper.selectAll()).thenReturn(announcements);

        // 调用控制器方法
        Result result = announcementController.listAll();

        // 验证结果
        assertEquals(200, result.getCode());
        assertEquals(announcements, result.getData().get("announcements"));

        // 验证Mapper调用
        verify(announcementMapper, times(1)).selectAll();
    }

    @Test
    void listLatest_ShouldReturnLatest5Announcements() {
        // 准备测试数据
        List<ANNOUNCEMENT> latestAnnouncements = Arrays.asList(mockAnnouncement);

        // 模拟Mapper行为
        when(announcementMapper.selectLatest(5)).thenReturn(latestAnnouncements);

        // 调用控制器方法
        Result result = announcementController.listLatest();

        // 验证结果
        assertEquals(200, result.getCode());
        assertEquals(latestAnnouncements, result.getData().get("announcements"));

        // 验证Mapper调用
        verify(announcementMapper, times(1)).selectLatest(5);
    }

    @Test
    void getDetail_ShouldReturnAnnouncement_WhenExists() {
        // 模拟Mapper行为
        when(announcementMapper.selectById(1)).thenReturn(mockAnnouncement);

        // 调用控制器方法
        Result result = announcementController.getDetail(1);

        // 验证结果
        assertEquals(200, result.getCode());
        assertEquals(mockAnnouncement, result.getData().get("announcement"));

        // 验证Mapper调用
        verify(announcementMapper, times(1)).selectById(1);
    }



    @Test
    void createAnnouncement_ShouldReturnSuccess_WhenValidInput() {
        // 模拟Mapper行为
        when(announcementMapper.insert(any(ANNOUNCEMENT.class))).thenReturn(1);

        // 调用控制器方法
        Result result = announcementController.createAnnouncement("测试标题", "测试内容");

        // 验证结果
        assertEquals(200, result.getCode());
        assertEquals("公告发布成功", result.getMessage());

        // 验证Mapper调用
        ArgumentCaptor<ANNOUNCEMENT> captor = ArgumentCaptor.forClass(ANNOUNCEMENT.class);
        verify(announcementMapper, times(1)).insert(captor.capture());

        ANNOUNCEMENT saved = captor.getValue();
        assertEquals("测试标题", saved.getTitle());
        assertEquals("测试内容", saved.getContent());
        assertNotNull(saved.getCreateTime());
    }

    @Test
    void createAnnouncement_ShouldReturnError_WhenInsertFails() {
        // 模拟Mapper行为
        when(announcementMapper.insert(any(ANNOUNCEMENT.class))).thenReturn(0);

        // 调用控制器方法
        Result result = announcementController.createAnnouncement("测试标题", "测试内容");

        // 验证结果
        assertEquals(500, result.getCode());
        assertEquals("公告发布失败", result.getMessage());

        // 验证Mapper调用
        verify(announcementMapper, times(1)).insert(any(ANNOUNCEMENT.class));
    }

    @Test
    void createAnnouncement_ShouldReturnError_WhenExceptionOccurs() {
        // 模拟Mapper抛出异常
        when(announcementMapper.insert(any(ANNOUNCEMENT.class)))
                .thenThrow(new RuntimeException("数据库错误"));

        // 调用控制器方法
        Result result = announcementController.createAnnouncement("测试标题", "测试内容");

        // 验证结果
        assertEquals(500, result.getCode());
        assertTrue(result.getMessage().contains("服务器错误"));

        // 验证Mapper调用
        verify(announcementMapper, times(1)).insert(any(ANNOUNCEMENT.class));
    }
}
