package com.ptp.ptplatform.controller;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.ptp.ptplatform.entity.HelpComment;
import com.ptp.ptplatform.service.HelpCommentService;
import com.ptp.ptplatform.service.HelpPostService;
import com.ptp.ptplatform.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.mock.web.MockMultipartFile;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

class HelpCommentControllerTest {

    @Mock
    private HelpCommentService commentService;

    @Mock
    private HelpPostService postService;

    @InjectMocks
    private HelpCommentController commentController;

    private final String authorId = "user123";
    private final String otherAuthorId = "user456";

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void likeComment_ShouldReturnSuccess() {
        int commentId = 1;
        doNothing().when(commentService).incrementLike(commentId);

        Result result = commentController.like(commentId);

        assertEquals(200, result.getCode());
        assertEquals("成功", result.getMessage());
        verify(commentService, times(1)).incrementLike(commentId);
    }

    @Test
    void getReplies_ShouldReturnReplies() {
        int commentId = 1;
        HelpComment r1 = new HelpComment();
        r1.setId(2); r1.setPostId(1); r1.setParentId(commentId);
        r1.setAuthorId("user2"); r1.setContent("Reply 1");
        r1.setCreateTime(LocalDateTime.now()); r1.setLikeCount(0);
        r1.setReplyTo("user1");

        HelpComment r2 = new HelpComment();
        r2.setId(3); r2.setPostId(1); r2.setParentId(commentId);
        r2.setAuthorId("user3"); r2.setContent("Reply 2");
        r2.setCreateTime(LocalDateTime.now()); r2.setLikeCount(0);
        r2.setReplyTo("user1");

        List<HelpComment> mockReplies = Arrays.asList(r1, r2);
        when(commentService.getReplies(commentId)).thenReturn(mockReplies);

        Result result = commentController.getReplies(commentId);

        assertEquals(200, result.getCode());
        @SuppressWarnings("unchecked")
        List<HelpComment> ret = (List<HelpComment>) result.getData().get("replies");
        assertEquals(mockReplies, ret);
        verify(commentService, times(1)).getReplies(commentId);
    }

    @Test
    void addReplyWithoutImage_ShouldReturnSuccess() {
        int commentId = 5;
        String authorId = "userX";
        String content = "no image reply";

        HelpComment parent = new HelpComment();
        parent.setId(commentId);
        parent.setAuthorId("userParent");
        parent.setPostId(10);
        when(commentService.getById(commentId)).thenReturn(parent);

        when(commentService.save(any(HelpComment.class))).thenReturn(true);

        Result result = commentController.addReply(commentId, authorId, content, null);

        assertEquals(200, result.getCode());
        HelpComment reply = (HelpComment) result.getData().get("reply");
        assertNotNull(reply);
        assertEquals(authorId, reply.getAuthorId());
        assertEquals(content, reply.getContent());
        assertNull(reply.getImageUrl());
        assertEquals(commentId, result.getData().get("parentCommentId"));
        verify(commentService, times(1)).save(any(HelpComment.class));
    }

    @Test
    void addReplyWithImage_ShouldReturnSuccess() {
        int commentId = 8;
        String authorId = "userY";
        String content = "with image reply";

        HelpComment parent = new HelpComment();
        parent.setId(commentId);
        parent.setAuthorId("userParent");
        parent.setPostId(20);
        when(commentService.getById(commentId)).thenReturn(parent);

        byte[] data = {0x1, 0x2};
        MockMultipartFile image = new MockMultipartFile(
                "image", "test.png", "image/png", data
        );

        when(commentService.save(any(HelpComment.class))).thenReturn(true);

        Result result = commentController.addReply(commentId, authorId, content, image);

        assertEquals(200, result.getCode());
        HelpComment reply = (HelpComment) result.getData().get("reply");
        assertNotNull(reply);
        assertEquals(authorId, reply.getAuthorId());
        assertEquals(content, reply.getContent());
        assertNotNull(reply.getImageUrl());
        assertTrue(reply.getImageUrl().startsWith("/uploads/"));
        assertEquals(commentId, result.getData().get("parentCommentId"));
        verify(commentService, times(1)).save(any(HelpComment.class));
    }

    @Test
    void addReply_ShouldReturnError_WhenParentNotExist() {
        int commentId = 99;
        when(commentService.getById(commentId)).thenReturn(null);

        Result result = commentController.addReply(commentId, "any", "content", null);

        assertEquals(500, result.getCode());
        assertEquals("被回复的评论不存在", result.getMessage());
        verify(commentService, never()).save(any());
    }

    @Test
    void deleteComment_ShouldReturnSuccess_WhenAuthorMatches() {
        // 准备测试数据
        int commentId = 1;
        HelpComment comment = new HelpComment();
        comment.setId(commentId);
        comment.setAuthorId(authorId);
        comment.setPostId(10);

        when(commentService.getById(commentId)).thenReturn(comment);
        when(commentService.removeById(commentId)).thenReturn(true);

        // 调用方法
        Result result = commentController.deleteComment(commentId, authorId);

        // 验证结果
        assertEquals(200, result.getCode());

        // 验证服务调用
        verify(commentService).removeById(commentId);
    }

}
