package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.INVITE_CODE;
import com.ptp.ptplatform.entity.USER;
import com.ptp.ptplatform.mapper.InviteCodeMapper;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.utils.Result;
import jakarta.servlet.http.HttpServletRequest;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.util.ArrayList;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class InviteCodeTest {

    @Mock
    private InviteCodeMapper inviteCodeMapper;

    @Mock
    private UserMapper userMapper;

    @Mock
    private UserController userController;

    @Mock
    private HttpServletRequest request;

    @InjectMocks
    private InviteCodeController inviteCodeController;

    private USER testUser;
    private INVITE_CODE testInviteCode;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);

        testUser = new USER();
        testUser.setUsername("testUser");
        testUser.setMagicPoints(15);

        testInviteCode = new INVITE_CODE(testUser.getUsername());
        testInviteCode.setCode("TESTCODE123");
    }

    @Test
    void generateInviteCode_Success() {
        // Arrange
        when(userController.getUserInRequest(request)).thenReturn(testUser);
        when(inviteCodeMapper.insertInviteCode(any(INVITE_CODE.class))).thenReturn(1);

        // Act
        Result result = inviteCodeController.generateInviteCode(request);

        // Assert
        assertTrue(result.isSuccess());
        assertNotNull(result.getData().get("inviteCode"));
        verify(userMapper, times(1)).updateUser(testUser);
    }

    @Test
    void generateInviteCode_Fail_NotEnoughMagicPoints() {
        // Arrange
        testUser.setMagicPoints(5);
        when(userController.getUserInRequest(request)).thenReturn(testUser);

        // Act
        Result result = inviteCodeController.generateInviteCode(request);

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("兑换邀请码失败，魔力值不足。", result.getMessage());
        verify(inviteCodeMapper, never()).insertInviteCode(any());
        verify(userMapper, never()).updateUser(any());
    }

    @Test
    void userInviteCode_Success() {
        // Arrange
        List<INVITE_CODE> inviteCodes = new ArrayList<>();
        inviteCodes.add(testInviteCode);

        when(userController.getUserInRequest(request)).thenReturn(testUser);
        when(inviteCodeMapper.selectByUser(testUser.getUsername())).thenReturn(inviteCodes);

        // Act
        Result result = inviteCodeController.userInviteCode(request);

        // Assert
        assertTrue(result.isSuccess());
        assertEquals(inviteCodes, result.getData().get("inviteCode"));
    }

    @Test
    void userInviteCode_EmptyList() {
        // Arrange
        when(userController.getUserInRequest(request)).thenReturn(testUser);
        when(inviteCodeMapper.selectByUser(testUser.getUsername())).thenReturn(new ArrayList<>());

        // Act
        Result result = inviteCodeController.userInviteCode(request);

        // Assert
        assertTrue(result.isSuccess());
        assertTrue(((List<?>) result.getData().get("inviteCode")).isEmpty());
    }
}