package com.ptp.ptplatform.controller;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ptp.ptplatform.entity.RequestComment;
import com.ptp.ptplatform.entity.RequestPost;
import com.ptp.ptplatform.entity.Notification;
import com.ptp.ptplatform.mapper.RequestPostMapper;
import com.ptp.ptplatform.service.RequestCommentService;
import com.ptp.ptplatform.service.RequestPostService;
import com.ptp.ptplatform.service.NotificationService;
import com.ptp.ptplatform.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.Mockito.*;

class RequestPostControllerTest {

    @Mock
    private RequestPostService postService;

    @Mock
    private RequestCommentService commentService;

    @Mock
    private NotificationService notificationService;

    @InjectMocks
    private RequestPostController postController;

    private final String authorId = "user123";
    private final String otherAuthorId = "user456";

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void createPost_ShouldReturnSuccess() throws Exception {
        // 准备测试数据
        String title = "Test Title";
        String content = "Test Content";
        MockMultipartFile image = new MockMultipartFile(
                "image", "test.jpg", "image/jpeg", "test image".getBytes());

        // 调用方法
        Result result = postController.createPost(title, content, authorId, image);

        // 验证结果
        assertEquals(200, result.getCode());
        RequestPost createdPost = (RequestPost) result.getData().get("post");
        assertNotNull(createdPost.getImageUrl());
        assertTrue(createdPost.getImageUrl().startsWith("/uploads/"));

        // 验证服务调用
        verify(postService).save(any(RequestPost.class));
    }

    @Test
    void getPost_ShouldReturnPostWithComments_WhenPostExists() {
        int postId = 1;
        RequestPost post = new RequestPost();
        post.setId(postId);
        post.setTitle("Test Post");
        post.setContent("Test Content");
        post.setAuthorId("user1");
        post.setCreateTime(LocalDateTime.now());

        RequestComment c1 = new RequestComment();
        c1.setId(1); c1.setPostId(postId); c1.setParentId(0);
        c1.setAuthorId("user2"); c1.setContent("Comment 1");
        c1.setCreateTime(LocalDateTime.now());

        RequestComment c2 = new RequestComment();
        c2.setId(2); c2.setPostId(postId); c2.setParentId(0);
        c2.setAuthorId("user3"); c2.setContent("Comment 2");
        c2.setCreateTime(LocalDateTime.now());

        List<RequestComment> comments = Arrays.asList(c1, c2);

        when(postService.getById(postId)).thenReturn(post);
        // 避免重载二义性，明确 stub list(Wrapper)
        doReturn(comments)
                .when(commentService)
                .list(any(Wrapper.class));

        Result result = postController.getPost(postId);

        assertEquals(200, result.getCode());
        assertEquals(post, result.getData().get("post"));
        @SuppressWarnings("unchecked")
        List<RequestComment> ret = (List<RequestComment>) result.getData().get("comments");
        assertEquals(2, ret.size());

        verify(postService, times(1)).getById(postId);
        verify(commentService, times(1)).list(any(Wrapper.class));
    }

    @Test
    void listPosts_ShouldReturnPaginatedPosts() {
        int page = 1, size = 5;
        RequestPost p1 = new RequestPost(); p1.setId(1); p1.setAuthorId("u1");
        RequestPost p2 = new RequestPost(); p2.setId(2); p2.setAuthorId("u2");
        List<RequestPost> posts = Arrays.asList(p1, p2);

        Page<RequestPost> pageResult = new Page<>(page, size);
        pageResult.setRecords(posts).setTotal(posts.size());

        when(postService.page(any(Page.class), any())).thenReturn(pageResult);

        Result result = postController.listPosts(page, size);

        assertEquals(200, result.getCode());
        assertEquals(posts, result.getData().get("records"));
        assertEquals((long) posts.size(), result.getData().get("total"));

        verify(postService, times(1)).page(any(Page.class), any());
    }

    @Test
    void getPost_ShouldReturnError_WhenPostNotExists() {
        int postId = 1;
        when(postService.getById(postId)).thenReturn(null);

        Result result = postController.getPost(postId);

        assertEquals(500, result.getCode());
        assertEquals("帖子不存在", result.getMessage());

        verify(postService, times(1)).getById(postId);
    }

    @Test
    void likePost_ShouldReturnSuccess() {
        int postId = 1;
        String likerId = "alice123";
        // 对应 ServiceImpl 中的 likePost(postId, likerId)
        doNothing().when(postService).likePost(postId, likerId);

        Result result = postController.likePost(postId, likerId);

        assertEquals(200, result.getCode());
        assertEquals("点赞成功", result.getMessage());
        verify(postService, times(1)).likePost(postId, likerId);
    }

    @Test
    void deletePost_ShouldReturnSuccess_WhenAuthorMatches() {
        int postId = 42;
        RequestPost mockPost = new RequestPost();
        mockPost.setAuthorId(authorId);
        // remove 返回 boolean，而非 void
        when(postService.getById(postId)).thenReturn(mockPost);
        when(commentService.remove(any(Wrapper.class))).thenReturn(true);
        when(postService.removeById(postId)).thenReturn(true);

        Result result = postController.deletePost(postId, authorId);

        assertEquals(200, result.getCode());
        verify(commentService, times(1)).remove(any(Wrapper.class));
        verify(postService, times(1)).removeById(postId);
    }

    @Test
    void addCommentWithoutImage_ShouldReturnSuccess() {
        int postId    = 7;
        String author = "user2";
        String content= "Hello";

        // 构造“保存后”返回给前端的 RequestComment 对象
        RequestComment saved = new RequestComment();
        saved.setId(99);
        saved.setPostId(postId);
        saved.setAuthorId(author);
        saved.setContent(content);
        saved.setImageUrl(null);

        // 1) 当 commentService.commentOnRequestPost(...) 被调用时，
        //    我们在 doAnswer 中给它传入的 HelpComment 对象设置 ID = 99
        doAnswer(invocation -> {
            RequestComment arg = invocation.getArgument(0);
            arg.setId(saved.getId());
            return null; // commentOnRequestPost 返回 void
        }).when(commentService).commentOnRequestPost(any(RequestComment.class));

        // 2) 后面 commentService.getById(99) 返回我们上面构造的 saved
        when(commentService.getById(saved.getId())).thenReturn(saved);

        // 模拟：commentOnHelpPost() 内部会调用 postService.incrementReplyCount(postId)
        // 但控制器本身并不直接调用 incrementReplyCount，所以测试里不再 verify 这一行。
        // 只要保证 postService.getById(postId) 能返回一个带有 replyCount 的 RequestPost 即可：
        RequestPost stubPost = new RequestPost();
        stubPost.setReplyCount(5);
        when(postService.getById(postId)).thenReturn(stubPost);

        // 调用控制器的 commentOnPost(...) 方法，第四个参数传 null（不带图片）
        Result result = postController.commentOnPost(postId, author, content, null);

        assertEquals(200, result.getCode());
        assertEquals(saved, result.getData().get("comment"));
        assertEquals(5,     result.getData().get("newReplyCount"));

        // 确保 commentService.commentOnHelpPost(...) 被调用一次
        verify(commentService, times(1)).commentOnRequestPost(any(RequestComment.class));
        // postService.incrementReplyCount(...) 由 commentOnRequestPost 内部调用，因此测试里无需 verify
    }


    @Test
    void addCommentWithImage_ShouldReturnSuccess() throws Exception {
        int postId    = 8;
        String author = "user3";
        String content= "With Image";

        // 准备一个 “模拟上传文件”
        MockMultipartFile mockFile = new MockMultipartFile(
                "image", "test.jpg", "image/jpeg", new byte[]{1,2,3}
        );

        // 构造“保存后”返回给前端的 RequestComment 对象
        RequestComment saved = new RequestComment();
        saved.setId(100);
        saved.setPostId(postId);
        saved.setAuthorId(author);
        saved.setContent(content);
        saved.setImageUrl("/uploads/whatever.jpg");

        // 当 commentService.commentOnRequestPost(...) 被调用时，给传入的对象设置 ID 与 imageUrl
        doAnswer(invocation -> {
            RequestComment arg = invocation.getArgument(0);
            arg.setId(saved.getId());
            arg.setImageUrl(saved.getImageUrl());
            return null; // commentOnRequestPost 返回 void
        }).when(commentService).commentOnRequestPost(any(RequestComment.class));

        // 后面 commentService.getById(100) 返回我们构造的 saved
        when(commentService.getById(saved.getId())).thenReturn(saved);

        // 同样，postService.getById(postId) 返回一个带 replyCount=10 的 HelpPost
        RequestPost stubPost = new RequestPost();
        stubPost.setReplyCount(10);
        when(postService.getById(postId)).thenReturn(stubPost);

        // 调用 commentOnPost，第四个参数传 mockFile（模拟有图片上传）
        Result result = postController.commentOnPost(postId, author, content, mockFile);

        assertEquals(200, result.getCode());
        assertEquals(saved, result.getData().get("comment"));
        assertEquals(10,    result.getData().get("newReplyCount"));

        // 验证真正传给 commentService.commentOnRequestPost(...) 的 HelpComment 对象中，
        // imageUrl 已经被设置成 "/uploads/whatever.jpg"
        ArgumentCaptor<RequestComment> captor = ArgumentCaptor.forClass(RequestComment.class);
        verify(commentService).commentOnRequestPost(captor.capture());
        RequestComment toSave = captor.getValue();
        assertEquals(saved.getImageUrl(), toSave.getImageUrl());

        // 同样无需 verify postService.incrementReplyCount(...)，因为该调用在 commentOnHelpPost 内部完成
    }
}
