package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.TorrentComment;
import com.ptp.ptplatform.service.TorrentCommentService;
import com.ptp.ptplatform.utils.Result;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class TORRENTCommentControllerTest {

    @Mock
    private TorrentCommentService commentService;

    @InjectMocks
    private TorrentCommentController commentController;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
    }

    @Test
    void likeComment_ShouldReturnSuccess() {
        // Arrange
        int commentId = 1;
        doNothing().when(commentService).incrementLike(commentId);

        // Act
        Result result = commentController.like(commentId);

        // Assert
        assertEquals(200, result.getCode());
        assertEquals("成功", result.getMessage());
        verify(commentService, times(1)).incrementLike(commentId);
    }

    @Test
    void getReplies_ShouldReturnReplies() {
        // Arrange
        int commentId = 1;
        List<TorrentComment> mockReplies = Arrays.asList(
                createComment(2, 1, commentId, "user2", "Reply 1", LocalDateTime.now(), 0, "user1"),
                createComment(3, 1, commentId, "user3", "Reply 2", LocalDateTime.now(), 0, "user1")
        );
        when(commentService.getReplies(commentId)).thenReturn(mockReplies);

        // Act
        Result result = commentController.getReplies(commentId);

        // Assert
        assertEquals(200, result.getCode());
        assertEquals(mockReplies, result.getData().get("replies"));
        verify(commentService, times(1)).getReplies(commentId);
    }

    @Test
    void addReply_ShouldReturnSuccess_WhenParentCommentExists() {
        // Arrange
        int commentId = 1;
        // 使用 createComment 创建父评论
        TorrentComment parentComment = createComment(
                commentId,         // id
                1,                // postId
                0,                // parentId (0表示是顶级评论)
                "user1",          // authorId
                "Parent comment", // content
                LocalDateTime.now(), // createTime
                5,                // likeCount
                null             // replyTo
        );

        // 创建回复评论
        TorrentComment reply = new TorrentComment();
        reply.setAuthorId("user2");
        reply.setContent("Test reply");

        when(commentService.getById(commentId)).thenReturn(parentComment);
        when(commentService.save(any(TorrentComment.class))).thenReturn(true);

        // Act
        Result result = commentController.addReply(commentId, reply);

        // Assert
        assertEquals(200, result.getCode());
        assertNotNull(result.getData().get("reply"));
        assertEquals(commentId, result.getData().get("parentCommentId"));
        verify(commentService, times(1)).save(any(TorrentComment.class));
    }

    @Test
    void addReply_ShouldReturnError_WhenParentCommentNotExists() {
        // Arrange
        int commentId = 1;
        TorrentComment reply = new TorrentComment();
        reply.setAuthorId("user2");
        reply.setContent("Test reply");

        when(commentService.getById(commentId)).thenReturn(null);

        // Act
        Result result = commentController.addReply(commentId, reply);

        // Assert
        assertEquals(500, result.getCode());
        assertEquals("被回复的评论不存在", result.getMessage());
        verify(commentService, never()).save(any(TorrentComment.class));
    }

    // 辅助方法
    private TorrentComment createComment(Integer id, Integer postId, Integer parentId,
                                         String authorId, String content, LocalDateTime createTime,
                                         Integer likeCount, String replyTo) {
        TorrentComment comment = new TorrentComment();
        comment.setId(id);
        comment.setPostId(postId);
        comment.setParentId(parentId);
        comment.setAuthorId(authorId);
        comment.setContent(content);
        comment.setCreateTime(createTime);
        comment.setLikeCount(likeCount);
        comment.setReplyTo(replyTo);
        return comment;
    }
}
