package com.ptp.ptplatform.controller;

import com.ptp.ptplatform.entity.INVITE_CODE;
import com.ptp.ptplatform.entity.USER;
import com.ptp.ptplatform.mapper.InviteCodeMapper;
import com.ptp.ptplatform.mapper.UserMapper;
import com.ptp.ptplatform.service.UserLevelService;
import com.ptp.ptplatform.utils.JwtUtils;
import com.ptp.ptplatform.utils.Result;
import io.jsonwebtoken.Claims;
import jakarta.servlet.http.HttpServletRequest;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.MockitoAnnotations;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.util.*;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

class UserControllerTest {
    private MockMvc mockMvc;

    @Mock
    private UserMapper userMapper;

    @Mock
    private InviteCodeMapper inviteCodeMapper;

    @Mock
    private HttpServletRequest request;

    @Mock
    private UserLevelService userLevelService;

    @InjectMocks
    private UserController userController;

    private USER testUser;
    private INVITE_CODE testInviteCode;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);

        testUser = new USER("testUser", "testPassword", USER.Authority.USER);
        testUser.setLastLogin(new Date());
        testUser.setUpload(1073741824L); // 1GB in bytes
        testUser.setDownload(536870912L); // 0.5GB in bytes

        testInviteCode = new INVITE_CODE("testUser");
        testInviteCode.setCode("TESTCODE123");
        testInviteCode.setIsUsed(false);

        mockMvc = MockMvcBuilders
                .standaloneSetup(userController)
                .build();
    }

    @Test
    void login_Success() {
        // Arrange
        when(userMapper.selectByUsername(testUser.getUsername())).thenReturn(testUser);

        // Act
        Result result = userController.login(testUser.getUsername(), testUser.getPassword());

        // Assert
        assertTrue(result.isSuccess());
        assertNotNull(result.getData().get("token"));
        verify(userMapper, times(1)).updateUser(testUser);
    }

    @Test
    void login_Fail_WrongPassword() {
        // Arrange
        when(userMapper.selectByUsername(testUser.getUsername())).thenReturn(testUser);

        // Act
        Result result = userController.login(testUser.getUsername(), "wrongPassword");

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("密码错误", result.getMessage());
        verify(userMapper, never()).updateUser(any());
    }

    @Test
    void login_Fail_UserNotFound() {
        // Arrange
        when(userMapper.selectByUsername("nonexistentUser")).thenReturn(null);

        // Act
        Result result = userController.login("nonexistentUser", "anyPassword");

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("用户不存在", result.getMessage());
    }

    @Test
    void regist_Success() {
        // Arrange
        when(userMapper.selectByUsername("newUser")).thenReturn(null);
        when(inviteCodeMapper.selectByCode("VALIDCODE")).thenReturn(testInviteCode);

        // Act
        Result result = userController.regist("newUser", "newPassword", "VALIDCODE");

        // Assert
        assertTrue(result.isSuccess());
        assertEquals("新建用户成功", result.getMessage());
        verify(userMapper, times(1)).insertUser(any(USER.class));
        verify(inviteCodeMapper, times(1)).updateCodeUser("VALIDCODE");
    }

    @Test
    void regist_Fail_UsernameExists() {
        // Arrange
        when(userMapper.selectByUsername(testUser.getUsername())).thenReturn(testUser);

        // Act
        Result result = userController.regist(testUser.getUsername(), "anyPassword", "anyCode");

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("用户名已存在，注册失败", result.getMessage());
        verify(userMapper, never()).insertUser(any());
    }

    @Test
    void regist_Fail_InvalidCode() {
        // Arrange
        when(userMapper.selectByUsername("newUser")).thenReturn(null);
        when(inviteCodeMapper.selectByCode("INVALIDCODE")).thenReturn(null);

        // Act
        Result result = userController.regist("newUser", "newPassword", "INVALIDCODE");

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("邀请码不存在，注册失败", result.getMessage());
        verify(userMapper, never()).insertUser(any());
    }

    @Test
    void regist_Fail_UsedCode() {
        // Arrange
        testInviteCode.setIsUsed(true);
        when(userMapper.selectByUsername("newUser")).thenReturn(null);
        when(inviteCodeMapper.selectByCode("USEDCODE")).thenReturn(testInviteCode);

        // Act
        Result result = userController.regist("newUser", "newPassword", "USEDCODE");

        // Assert
        assertFalse(result.isSuccess());
        assertEquals("邀请码已经被使用，注册失败", result.getMessage());
        verify(userMapper, never()).insertUser(any());
    }

    @Test
    public void testGetAllUser() {
        // Arrange
        List<USER> mockUsers = Arrays.asList(
                new USER("admin", "adminPass", USER.Authority.ADMIN),
                new USER("user1", "pass1", USER.Authority.USER),
                new USER("user2", "pass2", USER.Authority.USER)
        );
        when(userMapper.selectAllUsers()).thenReturn(mockUsers);

        // Act
        Result result = userController.getAllUser();

        // Assert
        assertEquals(200, result.getCode());
        assertNotNull(result.getData().get("data"));
        assertEquals(3, ((List<?>) result.getData().get("data")).size());
    }

    @Test
    public void testChangeAuthority() {
        // Arrange
        Map<String, String> authorityMap = new HashMap<>();
        authorityMap.put("changeUsername", "testUser");
        authorityMap.put("authority", "ADMIN");

        when(userMapper.selectByUsername("testUser")).thenReturn(testUser);
        when(userMapper.updateUser(any(USER.class))).thenReturn(1);

        // Act
        Result result = userController.changeAuthority(request, authorityMap);

        // Assert
        assertEquals(200, result.getCode());
        assertEquals("修改用户权限成功", result.getMessage());
    }

    @Test
    void whenPostRefreshLevel_thenServiceCalled_andReturnOk() throws Exception {
        mockMvc.perform(post("/user/refreshLevel"))
                .andExpect(status().isOk());

        verify(userLevelService).refreshAllUserLevels();
    }
}