import { api } from './auth'; // 导入实际使用的api实例
import MockAdapter from 'axios-mock-adapter';
import {
  getAllUsers,
  searchUsers,
  updateUserAuthority,
  getAllDiscounts,
  getCurrentDiscount,
  addDiscount,
  deleteDiscount
} from './administer';

describe('Administer API', () => {
  let mock;

  beforeAll(() => {
    mock = new MockAdapter(api); // 使用api实例而不是axios
    localStorage.setItem('token', 'test-token');
  });

  afterEach(() => {
    mock.reset();
  });

describe('getAllUsers', () => {
  it('should fetch all users successfully', async () => {
    const mockUsers = [
      {
        username: 'user1',
        authority: 'USER',
        registTime: '2023-01-01',
        lastLogin: '2023-05-01',
        upload: 1000,
        download: 500,
        shareRate: 2.0,
        magicPoints: 100
      }
    ];

    mock.onGet('/user/allUser').reply(200, {
      code: 200,
      data: { data: mockUsers }
    });

    const result = await getAllUsers();
    expect(result).toEqual(mockUsers);
  });

  it('should return empty array when no users', async () => {
    mock.onGet('/user/allUser').reply(200, {
      code: 200,
      data: { data: [] }
    });

    const result = await getAllUsers();
    expect(result).toEqual([]);
  });

  it('should handle error when fetching users', async () => {
    mock.onGet('/user/allUser').reply(500, {
      message: 'Request failed with status code 500'
    });

    await expect(getAllUsers()).rejects.toThrow('Request failed with status code 500');
  });
});

  describe('searchUsers', () => {
    it('should search users successfully', async () => {
      const mockUsers = [
        {
          username: 'user1',
          authority: 'USER'
        }
      ];

      mock.onGet('/user/searchUser', { params: { key: 'user' } })
        .reply(200, {
          code: 200,
          data: { data: mockUsers }
        });

      const result = await searchUsers('user');
      expect(result).toEqual(mockUsers);
    });

    it('should handle empty search key', async () => {
      const mockUsers = [
        {
          username: 'user1',
          authority: 'USER'
        }
      ];

      mock.onGet('/user/searchUser', { params: { key: '' } })
        .reply(200, {
          code: 200,
          data: { data: mockUsers }
        });

      const result = await searchUsers('');
      expect(result).toEqual(mockUsers);
    });
  });

  describe('updateUserAuthority', () => {
    it('should update user authority successfully', async () => {
      const username = 'user1';
      const authority = 'ADMIN';

      mock.onPut('/user/changeAuthority', {
        changeUsername: username,
        authority: authority
      }).reply(200, {
        code: 200,
        message: '修改用户权限成功'
      });

      const result = await updateUserAuthority(username, authority);
      expect(result.code).toBe(200);
    });
  });

  describe('getAllDiscounts', () => {
    it('should fetch all discounts successfully', async () => {
      const mockDiscounts = [
        {
          id: 1,
          name: '五一活动',
          discountType: 'FREE',
          startTime: '2023-05-01T00:00:00',
          endTime: '2023-05-07T23:59:59'
        }
      ];

      mock.onGet('/discount/all').reply(200, {
        code: 200,
        data: { data: mockDiscounts }
      });

      const result = await getAllDiscounts();
      expect(result).toEqual(mockDiscounts);
    });
  });

  describe('getCurrentDiscount', () => {
  it('should fetch current discount successfully', async () => {
    const mockDiscount = {
      id: 1,
      name: '当前活动',
      discountType: 'HALF',
      startTime: '2023-05-01T00:00:00',
      endTime: '2023-05-07T23:59:59'
    };

    mock.onGet('/discount/current').reply(200, {
      code: 200,
      data: { data: mockDiscount }
    });

    const result = await getCurrentDiscount();
    expect(result).toEqual(mockDiscount);
  });

  it('should return null when no current discount', async () => {
    mock.onGet('/discount/current').reply(200, {
      code: 200,
      message: '目前没有进行中的折扣',
      data: null
    });

    const result = await getCurrentDiscount();
    expect(result).toBeNull();
  });

  it('should handle error when fetching current discount', async () => {
    mock.onGet('/discount/current').reply(500);

    await expect(getCurrentDiscount()).rejects.toThrow();
  });
});

  describe('addDiscount', () => {
    it('should add new discount successfully', async () => {
      const newDiscount = {
        name: '新活动',
        discountType: 'DOUBLE',
        startTime: '2023-06-01T00:00:00',
        endTime: '2023-06-07T23:59:59'
      };

      mock.onPost('/discount/add', newDiscount).reply(200, {
        code: 200,
        data: { data: { id: 2, ...newDiscount } }
      });

      const result = await addDiscount(newDiscount);
      expect(result.id).toBe(2);
    });
  });

  describe('deleteDiscount', () => {
    it('should delete discount successfully', async () => {
      const discountId = 1;

      mock.onDelete(`/discount/delete/${discountId}`).reply(200, {
        code: 200,
        message: '删除成功'
      });

      const result = await deleteDiscount(discountId);
      expect(result).toBe(true);
    });

    it('should handle error when deleting discount', async () => {
      const discountId = 1;

      mock.onDelete(`/discount/delete/${discountId}`).reply(500, {
        message: 'Request failed with status code 500' // 改为实际会收到的错误消息
      });

      // 修改预期为实际会抛出的错误消息
      await expect(deleteDiscount(discountId)).rejects.toThrow('Request failed with status code 500');
    });
  });
});