import MockAdapter from 'axios-mock-adapter';
import { api } from './auth';
import { getAnnouncements, getLatestAnnouncements, getAnnouncementDetail,postAnnouncement } from './announcement';

describe('公告API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

describe('postAnnouncement - 发布公告', () => {
  it('应该成功发布公告', async () => {
    const mockData = {
      title: '测试公告',
      content: '测试内容'
    };
    
    const mockResponse = {
      code: 200,
      message: '公告发布成功',
      data: {
        id: 123,
        title: mockData.title,
        content: mockData.content,
        createTime: new Date().toISOString()
      }
    };
  

    mockAxios.onPost('/announcement/create').reply(200, mockResponse);

    const response = await postAnnouncement(mockData);
    expect(response.data).toEqual(mockResponse);
    expect(response.data.code).toBe(200);
    expect(response.data.data.title).toBe(mockData.title);
    expect(response.data.data.content).toBe(mockData.content);
  });
});

  describe('getAnnouncements - 获取所有公告', () => {
    it('应该成功获取公告列表', async () => {
  const mockAnnouncements = [
    { id: 1, title: '公告1', content: '内容1', createTime: '2023-01-01T00:00:00Z' },
    { id: 2, title: '公告2', content: '内容2', createTime: '2023-01-01T00:00:00Z' }
  ];
  
  mockAxios.onGet('/announcement/list').reply(200, {
    code: 200,
    data: {
      announcements: mockAnnouncements
    }
  });

  const announcements = await getAnnouncements();
  expect(announcements).toEqual(mockAnnouncements); // 检查返回的数组
  expect(announcements).toHaveLength(2);
});

    
  });

  describe('getLatestAnnouncements - 获取最新公告', () => {
    it('应该成功获取最新公告', async () => {
      const mockResponse = {
        code: 200,
        data: {
          announcements: [
            { id: 1, title: '最新公告', content: '最新内容' }
          ]
        }
      };
      
      mockAxios.onGet('/announcement/latest').reply(200, mockResponse);

      const response = await getLatestAnnouncements();
      expect(response.data).toEqual(mockResponse);
      expect(response.data.data.announcements).toHaveLength(1);
    });

    it('应该正确处理服务器错误', async () => {
      mockAxios.onGet('/announcement/latest').reply(500);
      
      await expect(getLatestAnnouncements()).rejects.toThrow();
    });
  });

  describe('getAnnouncementDetail - 获取公告详情', () => {
    it('应该成功获取公告详情', async () => {
      const announcementId = 123;
      const mockResponse = {
        code: 200,
        data: {
          announcement: {
            id: announcementId,
            title: '详细公告',
            content: '详细内容',
            createTime: '2023-01-01T00:00:00Z'
          }
        }
      };
      
      mockAxios.onGet(`/announcement/${announcementId}`).reply(200, mockResponse);

      const response = await getAnnouncementDetail(announcementId);
      expect(response.data).toEqual(mockResponse);
      expect(response.data.data.announcement.id).toBe(announcementId);
    });

    it('应该处理公告不存在的情况', async () => {
      const announcementId = 999;
      const mockResponse = {
        code: 404,
        message: '公告不存在'
      };
      
      mockAxios.onGet(`/announcement/${announcementId}`).reply(404, mockResponse);

      try {
        await getAnnouncementDetail(announcementId);
    } catch (error) {
        expect(error.response.status).toBe(404);
        expect(error.response.data.code).toBe(404);
        expect(error.response.data.message).toBe('公告不存在');
    }
    });

    it('应该正确处理网络错误', async () => {
      const announcementId = 456;
      mockAxios.onGet(`/announcement/${announcementId}`).networkError();
      
      await expect(getAnnouncementDetail(announcementId)).rejects.toThrow();
    });
  });
});