import MockAdapter from 'axios-mock-adapter';
import { api, login, register, getUserInfo, isAdmin } from './auth';

describe('auth API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
    localStorage.clear();
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('login', () => {
    it('should send login request with username and password', async () => {
      const mockResponse = {
        code: 200,
        data: { 
          token: 'mock-token',
          userInfo: { username: 'testuser' }
        },
        message: '登录成功'
      };
      
      mockAxios.onPost('/user/login', undefined, {
        params: { username: 'testuser', password: 'testpass' }
      }).reply(200, mockResponse);
    
      const response = await login('testuser', 'testpass');
      
      expect(response.data).toEqual(mockResponse);
      expect(localStorage.getItem('token')).toBe('mock-token');
    });

    it('should handle login failure', async () => {
      mockAxios.onPost('/user/login').reply(401, {
        code: 401,
        message: '登录失败'
      });
      
      await expect(login('wronguser', 'wrongpass')).rejects.toThrow();
    });
  });

  describe('register', () => {
    it('should send register request with username, password and code', async () => {
      const mockResponse = {
        code: 200,
        message: '注册成功'
      };
      
      mockAxios.onPost('/user/regist', undefined, {
        params: { username: 'newuser', password: 'newpass', code: 'invite123' }
      }).reply(200, mockResponse);

      const response = await register('newuser', 'newpass', 'invite123');
      
      expect(response.data).toEqual(mockResponse);
    });

    it('should handle registration failure', async () => {
      mockAxios.onPost('/user/regist').reply(400, {
        code: 400,
        message: '注册失败'
      });
      
      await expect(register('newuser', 'newpass', 'wrongcode')).rejects.toThrow();
    });
  });

  describe('getUserInfo', () => {
    it('should send request to get user info', async () => {
      const mockResponse = {
        code: 200,
        data: { username: 'testuser', authority: 'USER' }
      };
      
      mockAxios.onGet('/user/userInfo').reply(200, mockResponse);
  
      const response = await getUserInfo();
      
      expect(response).toEqual(mockResponse.data);
    });
  
    it('should handle unauthorized request', async () => {
      mockAxios.onGet('/user/userInfo').reply(401);
      
      await expect(getUserInfo()).rejects.toThrow('Request failed with status code 401');
    });
  });
  
  describe('isAdmin', () => {
    it('should return true when user is admin', async () => {
      const mockResponse = {
        code: 200,
        data: { username: 'admin', authority: 'ADMIN' }
      };
      
      mockAxios.onGet('/user/userInfo').reply(200, mockResponse);

      const result = await isAdmin();
      expect(result).toBe(true);
    });

    it('should return false when user is not admin', async () => {
      const mockResponse = {
        code: 200,
        data: { username: 'user', authority: 'USER' }
      };
      
      mockAxios.onGet('/user/userInfo').reply(200, mockResponse);

      const result = await isAdmin();
      expect(result).toBe(false);
    });

    it('should return false when request fails', async () => {
      mockAxios.onGet('/user/userInfo').reply(401);
      
      const result = await isAdmin();
      expect(result).toBe(false);
    });
  });

  describe('request interceptor', () => {
    it('should add Authorization header when token exists', async () => {
      localStorage.setItem('token', 'test-token');
      const mockResponse = { data: 'success' };
      
      mockAxios.onGet('/test').reply((config) => {
        expect(config.headers.Authorization).toBe('Bearer test-token');
        return [200, mockResponse];
      });

      const response = await api.get('/test');
      expect(response.data).toEqual(mockResponse);
    });

    it('should not add Authorization header when token does not exist', async () => {
      const mockResponse = { data: 'success' };
      
      mockAxios.onGet('/test').reply((config) => {
        expect(config.headers.Authorization).toBeUndefined();
        return [200, mockResponse];
      });

      const response = await api.get('/test');
      expect(response.data).toEqual(mockResponse);
    });
  });
});