import MockAdapter from 'axios-mock-adapter';
import { api } from './auth'; // 添加api导入
import { likeHelpPostComment, getHelpCommentReplies, addHelpCommentReply, deleteHelpComment } from './helpComment';

describe('求助帖评论API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('likeHelpPostComment - 点赞求助帖评论', () => {
    it('应该成功发送点赞请求', async () => {
      const mockResponse = { code: 200, message: '点赞成功' };
      mockAxios.onPost('/help/comments/123/like').reply(200, mockResponse);

      const response = await likeHelpPostComment('123');
      expect(response.data).toEqual(mockResponse);
    });
  });

  describe('getHelpCommentReplies - 获取评论回复', () => {
    it('应该返回正确的回复数据结构', async () => {
      const mockData = [{ id: '1', content: '回复内容' }];
      mockAxios.onGet('/help/comments/456/replies').reply(200, { code: 200, data: mockData });

      const response = await getHelpCommentReplies('456');
      expect(response.data.data).toEqual(mockData);
    });
  });

  describe('addHelpCommentReply - 添加评论回复', () => {
    it('应该正确发送回复内容（无图片）', async () => {
      const commentId = '789';
      const replyData = {
        authorId: 'user1',
        content: '测试回复'
      };
      
      mockAxios.onPost(`/help/comments/${commentId}/replies`).reply(config => {
        const data = config.data;
        expect(data.get('authorId')).toBe(replyData.authorId);
        expect(data.get('content')).toBe(replyData.content);
        expect(data.has('image')).toBe(false);
        return [200, { code: 200 }];
      });

      const response = await addHelpCommentReply(commentId, replyData);
      expect(response.status).toBe(200);
    });
    it('应该正确处理带图片的回复', async () => {
      const commentId = '789';
      const replyData = {
        authorId: 'user1',
        content: '测试回复',
        image: new File(['content'], 'reply.jpg')
      };
      
      mockAxios.onPost(`/help/comments/${commentId}/replies`).reply(config => {
        const data = config.data;
        expect(data.get('image')).toBeInstanceOf(File);
        return [200, { code: 200 }];
      });

      const response = await addHelpCommentReply(commentId, replyData);
      expect(response.status).toBe(200);
    });
  });
  describe('deleteHelpComment - 删除评论', () => {
    it('应该正确发送删除请求', async () => {
      const commentId = '101112';
      const authorId = 'user1';
      
      mockAxios.onDelete(`/help/comments/${commentId}`).reply(config => {
        expect(config.params).toEqual({ authorId });
        return [200, { code: 200 }];
      });

      const response = await deleteHelpComment(commentId, authorId);
      expect(response.status).toBe(200);
    });
  });
});