import MockAdapter from 'axios-mock-adapter';
import { api } from './auth'; // 添加api导入
import { createHelpPost, getHelpPosts, getHelpPostDetail, likeHelpPost, addHelpPostComment,deleteHelpPost,searchHelpPosts } from './helpPost';

describe('求助帖API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('createHelpPost - 创建求助帖', () => {
    it('应该正确发送无图片帖子数据', async () => {
      const postData = {
        title: '测试标题',
        content: '测试内容',
        authorId: 'user123'
      };
      // 使用函数匹配器来验证FormData内容
      mockAxios.onPost('/help/posts').reply(config => {
        const data = config.data;
        expect(data.get('title')).toBe(postData.title);
        expect(data.get('content')).toBe(postData.content);
        expect(data.get('authorId')).toBe(postData.authorId);
        expect(data.has('image')).toBe(false);
        return [201, { code: 201 }];
      });

      const response = await createHelpPost(postData.title, postData.content, postData.authorId);
      expect(response.status).toBe(201);
    });
  });
  it('应该正确处理带图片的帖子', async () => {
    const postData = {
      title: '测试标题',
      content: '测试内容',
      authorId: 'user123',
      selectedImage: new File(['content'], 'test.jpg')
    };
    
    mockAxios.onPost('/help/posts').reply(config => {
      const data = config.data;
      expect(data.get('image')).toBeInstanceOf(File);
      return [201, { code: 201 }];
    });

    const response = await createHelpPost(
      postData.title, 
      postData.content, 
      postData.authorId, 
      postData.selectedImage
    );
    expect(response.status).toBe(201);
  });


  describe('getHelpPosts - 获取求助帖列表', () => {
    it('应该支持分页参数', async () => {
      const page = 2, size = 10;
      mockAxios.onGet('/help/posts', { params: { page, size } }).reply(200, {
        code: 200,
        data: []
      });

      const response = await getHelpPosts(page, size);
      expect(response.status).toBe(200);
    });
  });

  describe('likeHelpPost - 点赞求助帖', () => {
    it('应该正确发送点赞请求', async () => {
      mockAxios.onPost('/help/posts/post123/like').reply(200, { code: 200 });
      const response = await likeHelpPost('post123');
      expect(response.status).toBe(200);
    });
  });

  
  describe('deleteHelpPost - 删除帖子', () => {
    it('应该正确发送删除请求', async () => {
      const postId = 'post789';
      const authorId = 'user1';
      
      mockAxios.onDelete(`/help/posts/${postId}`).reply(config => {
        expect(config.params).toEqual({ authorId });
        return [200, { code: 200 }];
      });

      const response = await deleteHelpPost(postId, authorId);
      expect(response.status).toBe(200);
    });
  });
  describe('searchHelpPosts - 搜索求助帖', () => {
      it('应该正确发送搜索请求', async () => {
        const keyword = '测试';
        const page = 1, size = 5;
        
        mockAxios.onGet('/help/posts/search', {
          params: { keyword, page, size }
        }).reply(200, {
          code: 200,
          data: {
            records: [],
            total: 0
          }
        });
  
        const response = await searchHelpPosts(keyword, page, size);
        expect(response.status).toBe(200);
        expect(response.data.data).toBeDefined();
      });
    });
});