// src/api/personal.js
import { api } from './auth';

/**
 * 获取用户信息
 * @returns {Promise<Object>} 用户信息对象
 */
export const getUserInfo = async () => {
  try {
    const response = await api.get('/user/userInfo');
    if (response.data.code === 200) {
        const userData = response.data.data;
        return {
            username: userData.username,
            level: userData.level,
            registTime: formatDate(userData.registTime),
            magicPoints: userData.magicPoints,
            upload: userData.upload, 
            download: userData.download, 
            shareRate: userData.shareRate.toFixed(2)
        };
    }
    throw new Error(response.data.message || '获取用户信息失败');
  } catch (error) {
    console.error('获取用户信息失败:', error);
    throw error;
  }
};

export const formatFileSize = (bytes) => {
    if (bytes < 1024) {
      return bytes + ' B';
    }
    const kb = bytes / 1024;
    if (kb < 1024) {
      return kb.toFixed(2) + ' KB';
    }
    const mb = kb / 1024;
    if (mb < 1024) {
      return mb.toFixed(2) + ' MB';
    }
    const gb = mb / 1024;
    return gb.toFixed(2) + ' GB';
  };
  
  
  export const getDownloadQuota = async () => {
    try {
      const response = await api.get('/user/allowDownload');
      if (response.data.code === 200) {
        const data = response.data.data;
        return {
          total: data.total,  // 已经是字节
          used: data.used,
          remaining: data.remaining
        };
      }
      throw new Error(response.data.message || '获取下载额度失败');
    } catch (error) {
      console.error('获取下载额度失败:', error);
      throw error;
    }
  };

// 修正后的时间格式化（正确处理时区）
const formatDate = (dateString) => {
    const date = new Date(dateString);
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    return `${year}-${month}-${day}`;
  };


  export const getDownloadProgress = async () => {
    try {
      const response = await api.get('/torrent/getProgress');
      if (response.data.code === 200) {
        return response.data.data.progresses;
      }
      throw new Error(response.data.message || '获取下载进度失败');
    } catch (error) {
      console.error('获取下载进度失败:', error);
      throw error;
    }
  };

  export const getUserTorrents = async (page = 1, size = 5) => {
    try {
      const response = await api.get('/torrent/get/torrentMyself', {
        params: { page, size }
      });
      if (response.data.code === 200) {
        const records = response.data.data.records.map(item => ({
          ...item.torrent,
          downloadCount: item.downloadCount,
          formattedSize: item.formattedSize
        }));
        return {
          records: records,
          total: response.data.data.total
        };
      }
      throw new Error(response.data.message || '获取上传记录失败');
    } catch (error) {
      console.error('获取上传记录失败:', error);
      throw error;
    }
  };
  

  export const deleteTorrent = async (id) => {
    try {
      const response = await api.delete(`/torrent/deleteTorrent/${id}`);
      if (response.data.code === 200) {
        return response.data;
      }
      throw new Error(response.data.message || '删除种子失败');
    } catch (error) {
      console.error('删除种子失败:', error);
      throw error;
    }
  };


  export const generateInviteCode = async () => {
    try {
      const response = await api.post('/invitecode/generate');
      if (response.data.code === 200) {
        return response.data.data.inviteCode;
      }
      throw new Error(response.data.message || '生成邀请码失败');
    } catch (error) {
      console.error('生成邀请码失败:', error);
      throw error;
    }
  };

  export const getUserInviteCodes = async () => {
    try {
      const response = await api.get('/invitecode/userInviteCode');
      if (response.data.code === 200) {
        return response.data.data.inviteCode;
      }
      throw new Error(response.data.message || '获取邀请码列表失败');
    } catch (error) {
      console.error('获取邀请码列表失败:', error);
      throw error;
    }
  };

  export const exchangeUpload = async (magicPoints) => {
    try {
      const response = await api.post('/user/exchangeUpload', {
        magicPoint: magicPoints
      });
      if (response.data.code === 200) {
        return response.data;
      }
      throw new Error(response.data.message || '兑换上传量失败');
    } catch (error) {
      console.error('兑换上传量失败:', error);
      throw error;
    }
  };

  export const updatePassword = async (oldPassword, newPassword) => {
    try {
      const response = await api.put('/user/password', {
        oldPassword,
        newPassword
      });
      if (response.data.code === 200) {
        return response.data;
      }
      throw new Error(response.data.message || '修改密码失败');
    } catch (error) {
      console.error('修改密码失败:', error);
      throw error;
    }
  };
