import MockAdapter from 'axios-mock-adapter';
import { api } from './auth';
import { 
  getRecommendations,
  markRecommendationShown,
  markRecommendationClicked
} from './recommend';

describe('推荐API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('getRecommendations - 获取推荐内容', () => {
    it('应该成功获取推荐列表', async () => {
      const mockData = [
        { id: '1', title: '推荐1' },
        { id: '2', title: '推荐2' }
      ];
      const limit = 5;
      
      mockAxios.onGet('/recommend/for-user', { params: { limit } })
        .reply(200, mockData);

      const response = await getRecommendations(limit);
      expect(response).toEqual(mockData);
    });

    it('应该在请求失败时抛出错误', async () => {
      mockAxios.onGet('/recommend/for-user').networkError();
      
      await expect(getRecommendations()).rejects.toThrow();
    });
  });

  describe('markRecommendationShown - 标记推荐为已显示', () => {
    it('应该成功发送标记请求', async () => {
      const torrentId = '123';
      mockAxios.onPost(`/recommend/mark-shown/${torrentId}`).reply(200);
      
      await expect(markRecommendationShown(torrentId)).resolves.not.toThrow();
    });

    it('应该在请求失败时不抛出错误', async () => {
      const torrentId = '456';
      mockAxios.onPost(`/recommend/mark-shown/${torrentId}`).networkError();
      
      await expect(markRecommendationShown(torrentId)).resolves.not.toThrow();
    });
  });

  describe('markRecommendationClicked - 标记推荐为已点击', () => {
    it('应该成功发送标记请求', async () => {
      const torrentId = '789';
      mockAxios.onPost(`/recommend/mark-clicked/${torrentId}`).reply(200);
      
      await expect(markRecommendationClicked(torrentId)).resolves.not.toThrow();
    });

    it('应该在请求失败时不抛出错误', async () => {
      const torrentId = '101';
      mockAxios.onPost(`/recommend/mark-clicked/${torrentId}`).networkError();
      
      await expect(markRecommendationClicked(torrentId)).resolves.not.toThrow();
    });
  });
});