import MockAdapter from 'axios-mock-adapter';
import { api } from './auth';
import {
  likeRequestPostComment,
  getRequestCommentReplies,
  addRequestCommentReply,
  deleteRequestComment
} from './requestComment';

describe('求种帖评论API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('likeRequestPostComment - 点赞求种帖评论', () => {
    it('应该成功发送点赞请求', async () => {
      const commentId = 'reqc123';
      const mockResponse = { code: 200, message: '点赞成功' };
      mockAxios.onPost(`/request/comments/${commentId}/like`).reply(200, mockResponse);

      const response = await likeRequestPostComment(commentId);
      expect(response.data).toEqual(mockResponse);
    });
  });

  describe('getRequestCommentReplies - 获取评论回复', () => {
    it('应该返回正确的回复数据结构', async () => {
      const commentId = 'reqc456';
      const mockData = [{ id: '1', content: '求种回复内容' }];
      mockAxios.onGet(`/request/comments/${commentId}/replies`)
        .reply(200, { code: 200, data: mockData });

      const response = await getRequestCommentReplies(commentId);
      expect(response.data.data).toEqual(mockData);
    });
  });

  describe('addRequestCommentReply - 添加评论回复', () => {
    it('应该正确发送回复内容（无图片）', async () => {
      const commentId = 'reqc789';
      const replyData = {
        authorId: 'user1',
        content: '测试求种回复'
      };
      
      mockAxios.onPost(`/request/comments/${commentId}/replies`).reply(config => {
        const data = config.data;
        expect(data.get('authorId')).toBe(replyData.authorId);
        expect(data.get('content')).toBe(replyData.content);
        expect(data.has('image')).toBe(false);
        return [200, { code: 200 }];
      });

      const response = await addRequestCommentReply(commentId, replyData);
      expect(response.status).toBe(200);
    });

    it('应该正确处理带图片的回复', async () => {
      const commentId = 'reqc789';
      const replyData = {
        authorId: 'user1',
        content: '测试求种回复',
        image: new File(['content'], 'reply.jpg')
      };
      
      mockAxios.onPost(`/request/comments/${commentId}/replies`).reply(config => {
        const data = config.data;
        expect(data.get('image')).toBeInstanceOf(File);
        return [200, { code: 200 }];
      });

      const response = await addRequestCommentReply(commentId, replyData);
      expect(response.status).toBe(200);
    });
  });

  describe('deleteRequestComment - 删除评论', () => {
    it('应该正确发送删除请求', async () => {
      const commentId = 'reqc101';
      const authorId = 'user1';
      
      mockAxios.onDelete(`/request/comments/${commentId}`).reply(config => {
        expect(config.params).toEqual({ authorId });
        return [200, { code: 200 }];
      });

      const response = await deleteRequestComment(commentId, authorId);
      expect(response.status).toBe(200);
    });
  });
});