import MockAdapter from 'axios-mock-adapter';
import { api } from './auth';
import {
  createRequestPost,
  getRequestPosts,
  getRequestPostDetail,
  likeRequestPost,
  addRequestPostComment,
  deleteRequestPost,
  searchRequestPosts
} from './requestPost';

describe('求种帖API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
  });

  afterEach(() => {
    mockAxios.restore();
  });

  describe('createRequestPost - 创建求种帖', () => {
    it('应该正确发送无图片帖子数据', async () => {
      const postData = {
        title: '求种测试标题',
        content: '求种测试内容',
        authorId: 'user123'
      };
      
      mockAxios.onPost('/request/posts').reply(config => {
        const data = config.data;
        expect(data.get('title')).toBe(postData.title);
        expect(data.get('content')).toBe(postData.content);
        expect(data.get('authorId')).toBe(postData.authorId);
        expect(data.has('image')).toBe(false);
        return [201, { code: 201 }];
      });

      const response = await createRequestPost(
        postData.title,
        postData.content,
        postData.authorId
      );
      expect(response.status).toBe(201);
    });

    it('应该正确处理带图片的帖子', async () => {
      const postData = {
        title: '求种测试标题',
        content: '求种测试内容',
        authorId: 'user123',
        selectedImage: new File(['content'], 'request.jpg')
      };
      
      mockAxios.onPost('/request/posts').reply(config => {
        const data = config.data;
        expect(data.get('image')).toBeInstanceOf(File);
        return [201, { code: 201 }];
      });

      const response = await createRequestPost(
        postData.title,
        postData.content,
        postData.authorId,
        postData.selectedImage
      );
      expect(response.status).toBe(201);
    });
  });

  describe('getRequestPosts - 获取求种帖列表', () => {
    it('应该支持分页参数', async () => {
      const page = 2, size = 10;
      mockAxios.onGet('/request/posts', { params: { page, size } })
        .reply(200, {
          code: 200,
          data: {
            records: [],
            total: 0
          }
        });

      const response = await getRequestPosts(page, size);
      expect(response.status).toBe(200);
      expect(response.data.data).toBeDefined();
    });
  });

  describe('getRequestPostDetail - 获取求种帖详情', () => {
    it('应该正确获取帖子详情', async () => {
      const postId = 'req123';
      mockAxios.onGet(`/request/posts/${postId}`)
        .reply(200, {
          code: 200,
          data: {
            post: {
              id: postId,
              title: '测试求种帖'
            }
          }
        });

      const response = await getRequestPostDetail(postId);
      expect(response.status).toBe(200);
      expect(response.data.data.post.id).toBe(postId);
    });
  });



  describe('deleteRequestPost - 删除求种帖', () => {
    it('应该正确发送删除请求', async () => {
      const postId = 'req101';
      const authorId = 'user1';
      
      mockAxios.onDelete(`/request/posts/${postId}`, {
        params: { authorId }
      }).reply(200, { code: 200 });

      const response = await deleteRequestPost(postId, authorId);
      expect(response.status).toBe(200);
    });
  });

  describe('searchRequestPosts - 搜索求种帖', () => {
    it('应该正确发送搜索请求', async () => {
      const keyword = '测试';
      const page = 1, size = 5;
      
      mockAxios.onGet('/request/posts/search', {
        params: { keyword, page, size }
      }).reply(200, {
        code: 200,
        data: {
          records: [],
          total: 0
        }
      });

      const response = await searchRequestPosts(keyword, page, size);
      expect(response.status).toBe(200);
      expect(response.data.data).toBeDefined();
    });
  });
});