import MockAdapter from 'axios-mock-adapter';
import { api } from './auth';
import {
  createTorrent,
  getTorrents,
  getTorrentDetail,
  likeTorrent,
  addTorrentComment,
  searchTorrents
} from './torrent';

describe('种子资源API', () => {
  let mockAxios;

  beforeEach(() => {
    mockAxios = new MockAdapter(api);
    localStorage.setItem('token', 'test-token');
  });

  afterEach(() => {
    mockAxios.restore();
    localStorage.clear();
  });


  describe('getTorrents - 获取种子列表', () => {
    it('应该发送带分页参数的请求', async () => {
      const mockResponse = {
        code: 200,
        data: {
          list: [{ id: 1, name: '种子1' }, { id: 2, name: '种子2' }],
          total: 2
        }
      };

      mockAxios.onGet('/torrent', { params: { page: 2, size: 10 } })
        .reply(200, mockResponse);

      const response = await getTorrents(2, 10);
      expect(response.data).toEqual(mockResponse);
    });
  });

  describe('getTorrentDetail - 获取种子详情', () => {
    it('应该规范化返回的数据结构', async () => {
      const mockResponse = {
        code: 200,
        data: {
          torrent: { id: '123', name: '测试种子' },
          comments: [{ id: '1', content: '评论1' }]
        }
      };

      mockAxios.onGet('/torrent/123').reply(200, mockResponse);

      const response = await getTorrentDetail('123');
      expect(response.data.data.torrent.name).toBe('测试种子');
      expect(response.data.data.comments).toHaveLength(1);
    });

    it('应该处理没有评论的情况', async () => {
      const mockResponse = {
        code: 200,
        data: {
          torrent: { id: '123', name: '测试种子' },
          comments: null
        }
      };

      mockAxios.onGet('/torrent/123').reply(200, mockResponse);

      const response = await getTorrentDetail('123');
      expect(response.data.data.comments).toEqual([]);
    });
  });

  describe('likeTorrent - 点赞种子', () => {
    it('应该发送点赞请求', async () => {
      const mockResponse = { code: 200, message: '点赞成功' };
      mockAxios.onPost('/torrent/123/like').reply(200, mockResponse);

      const response = await likeTorrent('123');
      expect(response.data).toEqual(mockResponse);
    });
  });

  describe('addTorrentComment - 添加种子评论', () => {
    it('应该发送评论数据', async () => {
      const commentData = { content: '测试评论' };
      const mockResponse = { code: 200, message: '评论成功' };

      mockAxios.onPost('/torrent/123/comments', commentData)
        .reply(200, mockResponse);

      const response = await addTorrentComment('123', commentData);
      expect(response.data).toEqual(mockResponse);
    });
  });

  describe('searchTorrents - 搜索种子', () => {
    it('应该发送带搜索关键词和分页的请求', async () => {
      const mockResponse = {
        code: 200,
        data: {
          list: [{ id: 1, name: '匹配的种子' }],
          total: 1
        }
      };

      mockAxios.onGet('/torrent/search', { 
        params: { keyword: '测试', page: 1, size: 5 } 
      }).reply(200, mockResponse);

      const response = await searchTorrents('测试');
      expect(response.data).toEqual(mockResponse);
    });
  });
});