// src/components/AuthForm.jsx
import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { login, register } from '../api/auth';
import './AuthForm.css';

const AuthForm = ({ onLoginSuccess }) => {
  const [isLogin, setIsLogin] = useState(true);
  const [formData, setFormData] = useState({
    username: '',
    password: '',
    code: ''
  });
  const [error, setError] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const navigate = useNavigate();

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    try {
      if (isLogin) {
            const response = await login(formData.username, formData.password);
            if (response.data.code === 200) {
              localStorage.setItem('token', response.data.data.token);
              localStorage.setItem('username', formData.username);
              onLoginSuccess?.();
              navigate('/dashboard');
            } else {
              setError(response.data.message || '登录失败');
            }
          } else {
            const response = await register(
              formData.username,
              formData.password,
              formData.code
            );
            if (response.data.code === 200) {
              setError('');
              setIsLogin(true);
              setFormData({ username: '', password: '', code: '' });
            } else {
              setError(response.data.message || '注册失败');
            }
          }
        } catch (err) {
          setError(err.message || (isLogin ? '登录失败' : '注册失败'));
        } finally {
          setIsLoading(false);
        }
      

      
  };

  return (
    <div className="auth-container">
      <h1 className="auth-title">PT影视资源站</h1>
      
      <div className="auth-form-wrapper">
        {isLogin ? (
          <form className="auth-form" onSubmit={handleSubmit}>
            <h2>用户登录</h2>
            {error && <div className="error-message">{error}</div>}
            
            <div className="form-group">
              <input
                type="text"
                name="username"
                placeholder="账号"
                value={formData.username}
                onChange={handleChange}
                required
              />
            </div>
            
            <div className="form-group">
              <input
                type="password"
                name="password"
                placeholder="密码"
                value={formData.password}
                onChange={handleChange}
                required
              />
            </div>
            
            <button type="submit" className="auth-button" disabled={isLoading}>
              {isLoading ? '登录中...' : '登录'}
            </button>
            
            <div className="auth-switch">
              <span>未有账户？</span>
              <button type="button" onClick={() => setIsLogin(false)} className="switch-button">
                去注册
              </button>
            </div>
          </form>
        ) : (
          <form className="auth-form" onSubmit={handleSubmit}>
            <h2>用户注册</h2>
            {error && <div className="error-message">{error}</div>}
            
            <div className="form-group">
              <input
                type="text"
                name="username"
                placeholder="账号"
                value={formData.username}
                onChange={handleChange}
                required
              />
            </div>
            
            <div className="form-group">
              <input
                type="password"
                name="password"
                placeholder="密码"
                value={formData.password}
                onChange={handleChange}
                required
              />
            </div>
            
            <div className="form-group">
              <input
                type="text"
                name="code"
                placeholder="邀请码"
                value={formData.code}
                onChange={handleChange}
                required
              />
            </div>
            
            <button type="submit" className="auth-button" disabled={isLoading}>
              {isLoading ? '注册中...' : '注册'}
            </button>
            
            <div className="auth-switch">
              <span>已有账户？</span>
              <button type="button" onClick={() => setIsLogin(true)} className="switch-button">
                去登录
              </button>
            </div>
          </form>
        )}
      </div>
    </div>
  );
};

export default AuthForm;