import { Modal, Form, Input, InputNumber, Select } from 'antd';
import React from 'react';
import { useIntl } from 'umi';
import { RewardItem } from '../data';
export type FormValueType = {
    rewardId?: number;
    title?: string;
    description?: string;
    amount?: number;
    status?: string;
    remark?: string;
} & Partial<RewardItem>;

export type UpdateFormProps = {
    onCancel: (flag?: boolean, formVals?: FormValueType) => void;
    onSubmit: (values: FormValueType) => Promise<void>;
    open: boolean;
    values: Partial<RewardItem>;
    statusOptions: any;
    readOnly?: boolean;
};

const UpdateForm: React.FC<UpdateFormProps> = (props) => {
    const [form] = Form.useForm();

    const { statusOptions } = props;

    const intl = useIntl();

    React.useEffect(() => {
        form.resetFields();
        form.setFieldsValue({
            rewardId: props.values.rewardId,
            title: props.values.title,
            description: props.values.description,
            amount: props.values.amount,
            status: props.values.status,
            remark: props.values.remark,
        });
    }, [form, props]);

    const handleOk = () => {
        form.submit();
    };
    const handleCancel = () => {
        props.onCancel();
        form.resetFields();
    };
    const handleFinish = async (values: FormValueType) => {
        props.onSubmit(values);
    };

    return (
        <Modal
            width={640}
            title={props.readOnly ? '查看悬赏' : (props.values.rewardId ? '编辑悬赏' : '新增悬赏')}
            open={props.open}
            onOk={handleOk}
            footer={props.readOnly ? null : undefined}
            onCancel={handleCancel}
        >
            <Form
                form={form}
                onFinish={handleFinish}
                initialValues={props.values}
                labelCol={{ span: 6 }}
                wrapperCol={{ span: 18 }}
            >
                <Form.Item name="rewardId" hidden={true}>
                    <Input />
                </Form.Item>
                <Form.Item
                    name="title"
                    label="悬赏标题"
                    rules={[{ required: true, message: '请输入悬赏标题！' }]}
                >
                    <Input placeholder="请输入悬赏标题" disabled={props.readOnly} />
                </Form.Item>
                <Form.Item
                    name="description"
                    label="悬赏描述"
                    rules={[{ required: true, message: '请输入悬赏描述！' }]}
                >
                    <Input.TextArea rows={4} placeholder="请输入悬赏描述" disabled={props.readOnly} />
                </Form.Item>
                <Form.Item
                    name="amount"
                    label="悬赏金额"
                    rules={[{ required: true, message: '请输入悬赏金额！' }]}
                >
                    <InputNumber
                        style={{ width: '100%' }}
                        min={0}
                        precision={2}
                        placeholder="请输入悬赏金额"
                        disabled={props.readOnly}
                    />
                </Form.Item>
                {/* <Form.Item
                    name="status"
                    label="悬赏状态"
                >
                    <Select
                        placeholder="请选择悬赏状态"
                        options={statusOptions.map((item: any) => ({
                            value: item.dictValue,
                            label: item.dictLabel,
                        }))}
                    />
                </Form.Item> */}
                <Form.Item
                    name="remark"
                    label="备注"
                >
                    <Input.TextArea rows={2} placeholder="请输入备注" disabled={props.readOnly} />
                </Form.Item>
            </Form>
        </Modal>
    );
};

export default UpdateForm;