import React, { useState, useEffect } from 'react';
import { useParams, useNavigate } from 'react-router-dom';
import { Button, Card, Form, Input, List, message, Avatar } from 'antd';
import { ArrowLeftOutlined } from '@ant-design/icons';
import { Layout } from 'antd';

const { Content } = Layout;
const { TextArea } = Input;

interface CommentItem {
    id: number;
    content: string;
    createTime: string;
    createBy: string;
    torrentId: number;
}

const TorrentComments: React.FC = () => {
    const { torrentId } = useParams<{ torrentId: string }>();
    const navigate = useNavigate();
    const [form] = Form.useForm();
    const [comments, setComments] = useState<CommentItem[]>([]);
    const [submitting, setSubmitting] = useState(false);

    // 获取评论列表
    const fetchComments = async () => {
        try {
            // 模拟评论数据
            const mockComments: CommentItem[] = [
                {
                    id: 1,
                    content: '这个种子非常好，下载速度很快！',
                    createTime: '2024-01-15 14:30:00',
                    createBy: '张三',
                    torrentId: Number(torrentId)
                },
                {
                    id: 2,
                    content: '资源质量很高，感谢分享。',
                    createTime: '2024-01-15 15:20:00',
                    createBy: '李四',
                    torrentId: Number(torrentId)
                },
                {
                    id: 3,
                    content: '这个版本很完整，推荐下载！',
                    createTime: '2024-01-15 16:45:00',
                    createBy: '王五',
                    torrentId: Number(torrentId)
                },
                {
                    id: 1,
                    content: '这个种子非常好，下载速度很快！',
                    createTime: '2024-01-15 14:30:00',
                    createBy: '张三',
                    torrentId: Number(torrentId)
                },
                {
                    id: 2,
                    content: '资源质量很高，感谢分享。',
                    createTime: '2024-01-15 15:20:00',
                    createBy: '李四',
                    torrentId: Number(torrentId)
                },
                {
                    id: 3,
                    content: '这个版本很完整，推荐下载！',
                    createTime: '2024-01-15 16:45:00',
                    createBy: '王五',
                    torrentId: Number(torrentId)
                }
            ];
            setComments(mockComments);
        } catch (error) {
            message.error('获取评论失败');
        }
    };

    useEffect(() => {
        if (torrentId) {
            fetchComments();
        }
    }, [torrentId]);

    // 提交评论
    const handleSubmit = async () => {
        try {
            const values = await form.validateFields();
            setSubmitting(true);

            // TODO: 替换为实际的API调用
            // await addComment({
            //   torrentId: Number(torrentId),
            //   content: values.content,
            // });

            message.success('评论成功');
            form.resetFields();
            fetchComments(); // 刷新评论列表
        } catch (error) {
            message.error('评论失败');
        } finally {
            setSubmitting(false);
        }
    };

    return (
        <Content style={{
            height: '100vh',
            display: 'flex',
            flexDirection: 'column',
            overflow: 'hidden' // 防止内容溢出
        }}>
            {/* 顶部标题栏 */}
            <div style={{
                padding: '16px',
                borderBottom: '1px solid #f0f0f0',
                display: 'flex',
                alignItems: 'center',
                backgroundColor: '#fff',
                zIndex: 10
            }}>
                <Button
                    type="link"
                    icon={<ArrowLeftOutlined />}
                    onClick={() => navigate(-1)}
                    style={{ marginRight: '10px', padding: 0 }}
                />
                <span style={{ fontSize: '16px', fontWeight: 'bold' }}>种子评论</span>
            </div>

            {/* 评论列表区域 - 可滚动 */}
            <div style={{
                flex: 1,
                overflowY: 'auto',
                padding: '0 16px',
                paddingBottom: '16px'
            }}>
                <List
                    className="comment-list"
                    itemLayout="horizontal"
                    dataSource={comments}
                    renderItem={(item) => (
                        <List.Item>
                            <List.Item.Meta
                                avatar={<Avatar>{item.createBy[0]}</Avatar>}
                                title={item.createBy}
                                description={
                                    <div>
                                        <div>{item.content}</div>
                                        <div style={{ color: '#8c8c8c', fontSize: '12px', marginTop: '8px' }}>
                                            {item.createTime}
                                        </div>
                                    </div>
                                }
                            />
                        </List.Item>
                    )}
                />
            </div>

            {/* 评论输入框 - 固定在父组件底部 */}
            <div style={{
                position: 'relative',
                padding: '16px',
                backgroundColor: '#fff',
                borderTop: '1px solid #f0f0f0',
                boxShadow: '0 -2px 8px rgba(0, 0, 0, 0.06)'
            }}>
                <Form form={form}>
                    <Form.Item
                        name="content"
                        rules={[{ required: true, message: '请输入评论内容' }]}
                        style={{ marginBottom: '12px' }}
                    >
                        <TextArea rows={3} placeholder="请输入您的评论" />
                    </Form.Item>
                    <Form.Item style={{ marginBottom: 0, textAlign: 'right' }}>
                        <Button
                            htmlType="submit"
                            loading={submitting}
                            onClick={handleSubmit}
                            type="primary"
                        >
                            提交评论
                        </Button>
                    </Form.Item>
                </Form>
            </div>
        </Content>
    );
};

export default TorrentComments;