package com.ruoyi.announce.controller;

import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.announce.service.IAnnounceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;

@RestController
public class AnnounceController extends BaseController {

    @Autowired
    private IAnnounceService announceService;

    /**
     * BT Tracker /announce 接口
     * 接收客户端（qBittorrent 等）发来的 announce 请求，返回 bencoded peers 列表
     */
    @GetMapping(value = "/announce", produces = "application/x-bittorrent")
    public void announce(
            HttpServletRequest request,
            HttpServletResponse response,
            @RequestParam("info_hash") String infoHashParam,
            @RequestParam("peer_id")   String peerIdParam,
            @RequestParam("port")      int port,
            @RequestParam("uploaded")  long uploaded,
            @RequestParam("downloaded") long downloaded,
            @RequestParam("left")      long left,
            @RequestParam(value = "event",    required = false) String event,
            @RequestParam("passkey")   String passkey
    ) throws Exception {
        // 1. URL Decode 得到原始二进制
        byte[] infoHash = URLDecoder.decode(infoHashParam, StandardCharsets.ISO_8859_1.name())
                .getBytes(StandardCharsets.ISO_8859_1);
        byte[] peerId   = URLDecoder.decode(peerIdParam,   StandardCharsets.ISO_8859_1.name())
                .getBytes(StandardCharsets.ISO_8859_1);

        // 2. 处理 announce 请求（验证 passkey，更新 peer 列表，获取 peers 信息）
        Map<String, Object> reply = announceService.handleAnnounce(
                infoHash, peerId, port, uploaded, downloaded, left, event, passkey,
                request.getRemoteAddr()
        );

        // 3. bencode 编码并返回给客户端
        response.setStatus(HttpServletResponse.SC_OK);
        response.setHeader("Content-Type", "application/x-bittorrent");
        try (var out = response.getOutputStream()) {
            byte[] bencoded = announceService.encodeBencode(reply);
            out.write(bencoded);
            out.flush();
        }
    }
}
