package com.ruoyi.announce.service.impl;

import com.ruoyi.announce.service.IAnnounceService;
import com.ruoyi.authentication.domain.SysUserPasskey;
import com.ruoyi.authentication.domain.UserScore;
import com.ruoyi.authentication.service.ISysUserPasskeyService;
import com.ruoyi.authentication.service.IUserScoreService;
import com.ruoyi.common.core.redis.RedisCache;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.TimeUnit;

import static java.lang.Long.parseLong;

@Service
public class AnnounceServiceImpl implements IAnnounceService {

    private static final int ANNOUNCE_INTERVAL = 1800;
    private static final int PEER_CACHE_DURATION_MIN = 60;
    private static final int MAX_SEEDERS = 25;
    private static final int MAX_LEECHERS = 50;
    private static final int UPLOAD_WEIGHT = 70;
    private static final int DOWNLOAD_WEIGHT = 30;

    @Autowired
    private RedisCache redisCache;
    @Autowired
    private ISysUserPasskeyService passkeyService;
    @Autowired
    private IUserScoreService userScoreService;

    @Override
    public Map<String, Object> handleAnnounce(byte[] infoHash,
                                              byte[] peerId,
                                              int port,
                                              long uploaded,
                                              long downloaded,
                                              long left,
                                              String event,
                                              String passkey,
                                              String ip) throws IOException {
        if (passkey == null) {
            return Collections.emptyMap();
        }

        String infoHashHex = toHex(infoHash);
        String peerKey = buildPeerKey(peerId);
        String seederSet = buildZSetKey(infoHashHex, "seeders");
        String leecherSet = buildZSetKey(infoHashHex, "leechers");

        if ("stopped".equals(event)) {
            return handleStopEvent(peerKey, seederSet, leecherSet, passkey, uploaded, downloaded);
        }

        cachePeerState(peerKey, uploaded, downloaded, left, ip, port);
        updatePeerRanking(seederSet, leecherSet, peerKey, left == 0);

        List<byte[]> peerList = selectPeers(seederSet, leecherSet, peerKey, left == 0);
        byte[] peersBinary = flattenPeers(peerList);

        return buildResponse(seederSet, leecherSet, peersBinary);
    }

    private Map<String, Object> handleStopEvent(String peerKey,
                                                String seederSet,
                                                String leecherSet,
                                                String passkey,
                                                long uploaded,
                                                long downloaded) {
        redisCache.deleteObject(peerKey);
        redisCache.zRemove(seederSet, peerKey);
        redisCache.zRemove(leecherSet, peerKey);

        Optional<Long> userId = resolveUserId(passkey);
        userId.ifPresent(id -> updateUserScore(id, uploaded, downloaded));
        return Collections.emptyMap();
    }

    private void cachePeerState(String peerKey,
                                long uploaded,
                                long downloaded,
                                long left,
                                String ip,
                                int port) {
        Map<String, String> data = new HashMap<>();
        data.put("ip", ip);
        data.put("port", String.valueOf(port));
        data.put("uploaded", String.valueOf(uploaded));
        data.put("downloaded", String.valueOf(downloaded));
        data.put("left", String.valueOf(left));
        data.put("lastSeen", String.valueOf(System.currentTimeMillis()));

        redisCache.setCacheMap(peerKey, data);
        redisCache.expire(peerKey, PEER_CACHE_DURATION_MIN, TimeUnit.MINUTES);
    }

    private void updatePeerRanking(String seederSet,
                                   String leecherSet,
                                   String peerKey,
                                   boolean isSeeder) {
        String sourceSet = isSeeder ? leecherSet : seederSet;
        String targetSet = isSeeder ? seederSet : leecherSet;

        redisCache.zRemove(sourceSet, peerKey);
        redisCache.zAdd(targetSet, peerKey, System.currentTimeMillis());
    }

    private List<byte[]> selectPeers(String seederSet,
                                     String leecherSet,
                                     String selfKey,
                                     boolean isSeeder) throws IOException {
        Set<String> candidates = new LinkedHashSet<>();
        if (isSeeder) {
            candidates.addAll(redisCache.zRange(leecherSet, 0, MAX_LEECHERS));
        } else {
            candidates.addAll(Optional.ofNullable(redisCache.zRange(seederSet, 0, MAX_SEEDERS)).orElse(Collections.emptySet()));
            candidates.addAll(Optional.ofNullable(redisCache.zRange(leecherSet, 0, MAX_LEECHERS)).orElse(Collections.emptySet()));
        }

        List<byte[]> peers = new ArrayList<>();
        for (String candidateKey : candidates) {
            if (candidateKey.equals(selfKey)) continue;
            cachePeerLevel:
            {
                Map<String, String> info = redisCache.getCacheMap(candidateKey);
                String peerIp = info.get("ip");
                String portStr = info.get("port");
                if (peerIp == null || peerIp.contains(":") || portStr == null) break cachePeerLevel;
                peers.add(encodePeer(peerIp, Integer.parseInt(portStr)));
            }
        }
        return peers;
    }

    private byte[] flattenPeers(List<byte[]> peerList) throws IOException {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            for (byte[] peer : peerList) {
                out.write(peer);
            }
            return out.toByteArray();
        }
    }

    private Map<String, Object> buildResponse(String seederSet,
                                              String leecherSet,
                                              byte[] peersBinary) {
        Map<String, Object> response = new LinkedHashMap<>();
        response.put("interval", ANNOUNCE_INTERVAL);
        response.put("min interval", ANNOUNCE_INTERVAL / 2);
        response.put("complete", Optional.ofNullable(redisCache.zCard(seederSet)).orElse(0L).intValue());
        response.put("incomplete", Optional.ofNullable(redisCache.zCard(leecherSet)).orElse(0L).intValue());
        response.put("peers", peersBinary);
        return response;
    }

    private Optional<Long> resolveUserId(String passkey) {
        SysUserPasskey criteria = new SysUserPasskey(passkey);
        List<SysUserPasskey> records = passkeyService.selectSysUserPasskeyList(criteria);
        return records.isEmpty() ? Optional.empty() : Optional.of(records.get(0).getUserId());
    }

    private void updateUserScore(Long userId, long uploaded, long downloaded) {
        try {
            UserScore existing = userScoreService.selectUserScoreByUserId(userId);
            boolean isNew = existing.getUserId() == null;
            long totalUploaded = isNew ? uploaded : parseLong(existing.getUploaded()) + uploaded;
            long totalDownloaded = isNew ? downloaded : parseLong(existing.getDownloaded()) + downloaded;
            long combinedScore = calculateWeightedScore(totalUploaded, totalDownloaded);

            existing.setUserId(userId);
            existing.setUploaded(String.valueOf(totalUploaded));
            existing.setDownloaded(String.valueOf(totalDownloaded));
            existing.setScore(combinedScore);
            existing.setUpdatedAt(new Date());

            if (isNew) {
                userScoreService.insertUserScore(existing);
            } else {
                userScoreService.updateUserScore(existing);
            }
        } catch (Exception e) {
            // Log exception properly in production code
            System.err.println("Failed to update user score: " + e.getMessage());
        }
    }

    private long calculateWeightedScore(long uploaded, long downloaded) {
        return (uploaded * UPLOAD_WEIGHT + downloaded * DOWNLOAD_WEIGHT) / (UPLOAD_WEIGHT + DOWNLOAD_WEIGHT);
    }

    private String buildPeerKey(byte[] peerId) {
        return "peer:" + new String(peerId, StandardCharsets.ISO_8859_1);
    }

    private String buildZSetKey(String infoHashHex, String type) {
        return String.format("torrent:%s:%s", infoHashHex, type);
    }

    private String toHex(byte[] data) {
        char[] hex = "0123456789abcdef".toCharArray();
        StringBuilder sb = new StringBuilder(data.length * 2);
        for (byte b : data) {
            sb.append(hex[(b >> 4) & 0xF]).append(hex[b & 0xF]);
        }
        return sb.toString();
    }

    /**
     * Encode peer address into 6-byte compact representation as per BEP 23
     */
    private byte[] encodePeer(String ip, int port) {
        String[] parts = ip.split("\\.");
        if (parts.length != 4) return null;
        ByteBuffer buffer = ByteBuffer.allocate(6);
        try {
            for (String part : parts) {
                int segment = Integer.parseInt(part);
                if (segment < 0 || segment > 255) return null;
                buffer.put((byte) segment);
            }
            buffer.putShort((short) port);
            return buffer.array();
        } catch (NumberFormatException e) {
            return null;
        }
    }

    @Override
    public Map<String, Object> handleScrape(String[] infoHashes) {
        Map<String, Object> stats = new HashMap<>();
        for (String hash : infoHashes) {
            String key = hash.toLowerCase();
            Map<String, Integer> countMap = Map.of(
                    "complete", getZCardAsInt(buildZSetKey(key, "seeders")),
                    "incomplete", getZCardAsInt(buildZSetKey(key, "leechers")),
                    "downloaded", 0
            );
            stats.put(hash, countMap);
        }
        return stats;
    }

    /**
     * 查询当前所有活跃的 Peer 节点，包含详细字段
     */
    @Override
    public Map<String, Object> handleHeartbeat( String ip) {
        // 延长该 IP 下所有 peer 的缓存有效期
        Collection<String> keys = redisCache.keys("peer:*");
        for (String key : keys) {
            Map<String, String> data = redisCache.getCacheMap(key);
            if (data == null) continue;
            String peerIp = data.get("ip");
            if (ip.equals(peerIp)) {
                redisCache.expire(key, PEER_CACHE_DURATION_MIN, TimeUnit.MINUTES);
            }
        }

        // 构造基本响应（可根据需要扩展）
        return Map.of(
                "interval", ANNOUNCE_INTERVAL,
                "min interval", ANNOUNCE_INTERVAL / 2,
                "complete", Optional.ofNullable(redisCache.zCard("some:default:torrent:seeders")).orElse(0L).intValue(),
                "incomplete", Optional.ofNullable(redisCache.zCard("some:default:torrent:leechers")).orElse(0L).intValue(),
                "peers", new byte[0]
        );
    }

    /**
     * 查询当前所有活跃的 Peer 节点，包含详细字段
     */
    @Override
    public List<Map<String, Object>> getAllPeers() {
        // RedisCache.keys 返回 Collection<String>
        Collection<String> keys = redisCache.keys("peer:*");
        List<Map<String, Object>> peers = new ArrayList<>();
        for (String key : keys) {
            Map<String, String> data = redisCache.getCacheMap(key);
            if (data == null || data.isEmpty()) continue;
            Map<String, Object> info = new LinkedHashMap<>();
            info.put("peerKey", key);
            info.put("ip", data.get("ip"));
            info.put("port", data.get("port"));
            info.put("uploaded", data.get("uploaded"));
            info.put("downloaded", data.get("downloaded"));
            info.put("left", data.get("left"));
            info.put("lastSeen", data.get("lastSeen"));
            peers.add(info);
        }
        return peers;
    }

    private void refreshPeersForIp(String ip) {
        redisCache.keys("peer:*").stream()
                .filter(key -> ip.equals(redisCache.getCacheMap(key).get("ip")))
                .forEach(key -> redisCache.expire(key, PEER_CACHE_DURATION_MIN, TimeUnit.MINUTES));
    }

    private List<byte[]> selectPeerBytes(Set<String> peers) {
        List<byte[]> list = new ArrayList<>();
        for (String peerId : peers) {
            Map<String, String> data = redisCache.getCacheMap("peer:" + peerId);
            if (data == null) continue;
            String ip = data.get("ip");
            String port = data.get("port");
            if (ip == null || ip.contains(":")) continue;
            list.add(encodePeer(ip, Integer.parseInt(port)));
        }
        return list;
    }

    private Map<String, Object> buildResponse(int incomplete, byte[] peers) {
        Map<String, Object> response = new LinkedHashMap<>();
        response.put("interval", ANNOUNCE_INTERVAL);
        response.put("min interval", ANNOUNCE_INTERVAL / 2);
        response.put("complete", 0);
        response.put("incomplete", incomplete);
        response.put("peers", peers);
        return response;
    }

    private int getZCardAsInt(String zsetKey) {
        Long count = redisCache.zCard(zsetKey);
        return Optional.ofNullable(count).orElse(0L).intValue();
    }
}