package com.ruoyi.torrent.controller;

import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import com.ruoyi.common.utils.http.HttpUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.torrent.domain.BtTorrentAnnounce;
import com.ruoyi.torrent.domain.BtTorrentFile;
import com.ruoyi.torrent.service.IBtTorrentAnnounceService;
import com.ruoyi.torrent.service.IBtTorrentFileService;
import com.ruoyi.torrent.service.IBtTorrentService;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.multipart.MultipartFile;

import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.torrent.domain.BtTorrent;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.UUID;

/**
 * 种子主Controller
 * 
 * @author ruoyi
 * @date 2025-04-21
 */
@RestController
@RequestMapping("/system/torrent")
public class BtTorrentController extends BaseController
{
    @Autowired
    private IBtTorrentService btTorrentService;

    @Autowired
    private IBtTorrentFileService btTorrentFileService;
    @Autowired
    private IBtTorrentAnnounceService btTorrentAnnounceService;
    private static final String RECOMMEND_API = "http://127.0.0.1:5000/recommend_torrents";

    private static final String boundary = "----WebKitFormBoundary" + UUID.randomUUID().toString().replace("-", "");

    @PreAuthorize("@ss.hasPermi('system:torrent:add')")
    @Log(title = "种子主", businessType = BusinessType.INSERT)
    @PostMapping("/uploadTorrent")
    public AjaxResult uploadTorrent(@RequestParam("file") MultipartFile file) {
        try {
            // Create URL connection to Flask server
            String flaskUrl = "http://localhost:5000/parse_torrent"; // Flask server URL
            HttpURLConnection connection = (HttpURLConnection) new URL(flaskUrl).openConnection();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);

            // Open output stream
            try (OutputStream outputStream = connection.getOutputStream()) {
                // Write the multipart form data boundary and file data
                writeFormData(outputStream, file);

                // Handle Flask response
                int responseCode = connection.getResponseCode();
                if (responseCode != 200) {
                    return AjaxResult.error("Failed to communicate with Flask server, response code: " + responseCode);
                }

                // Assuming the Flask server responds with JSON, parse the response
                String responseBody = new String(connection.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
                parseTorrentData(responseBody);
                return AjaxResult.success("Torrent uploaded and processed successfully");
            }
        } catch (IOException e) {
            e.printStackTrace();
            return AjaxResult.error("Error during file upload or communication with Flask service");
        }
    }

    // Write multipart form data to the output stream
    private void writeFormData(OutputStream outputStream, MultipartFile file) throws IOException {
        // Write the start boundary
        outputStream.write(("--" + boundary + "\r\n").getBytes(StandardCharsets.UTF_8));

        // Write content-disposition header
        outputStream.write(("Content-Disposition: form-data; name=\"torrent\"; filename=\"" + file.getOriginalFilename() + "\"\r\n").getBytes(StandardCharsets.UTF_8));
        outputStream.write(("Content-Type: application/octet-stream\r\n\r\n").getBytes(StandardCharsets.UTF_8));

        // Write the file content
        file.getInputStream().transferTo(outputStream);
        outputStream.write("\r\n".getBytes(StandardCharsets.UTF_8));

        // Write the closing boundary
        outputStream.write(("--" + boundary + "--\r\n").getBytes(StandardCharsets.UTF_8));
    }


    public void parseTorrentData(String responseData) {
        try {

            // Create ObjectMapper to handle JSON parsing
            ObjectMapper objectMapper = new ObjectMapper();

            // Parse the JSON response
            JsonNode jsonNode = objectMapper.readTree(responseData);
            JsonNode btTorrentNode = jsonNode.get("btTorrent");
            JsonNode btTorrentAnnounceNode = jsonNode.get("btTorrentAnnounce");
            JsonNode btTorrentFilesNode = jsonNode.get("btTorrentFiles");


            // Convert btTorrentNode to BtTorrent object
            BtTorrent btTorrent = objectMapper.readValue(btTorrentNode.toString(), BtTorrent.class);
            btTorrent.setCreatedBy(SecurityUtils.getUsername());
            btTorrent.setUploaderId(SecurityUtils.getUserId());
            btTorrentService.insertBtTorrent(btTorrent);
            Long torrentId=btTorrent.getTorrentId();

            // Convert btTorrentFilesNode to List<BtTorrentFile> using TypeReference
            List<BtTorrentFile> btTorrentFiles = objectMapper.readValue(
                    btTorrentFilesNode.toString(),
                    new TypeReference<List<BtTorrentFile>>() {});
            btTorrentFiles.forEach(btFile -> {
                btFile.setTorrentId(torrentId);
                btTorrentFileService.insertBtTorrentFile(btFile);
            }
            );

            // Convert btTorrentAnnounceNode to List<BtTorrentAnnounce>
            List<BtTorrentAnnounce> btTorrentAnnounceList = new ArrayList<>();
            if (btTorrentAnnounceNode.isArray()) {
                for (JsonNode announceNode : btTorrentAnnounceNode) {
                    // You may need to adjust this depending on your BtTorrentAnnounce class structure
                    BtTorrentAnnounce announce = objectMapper.readValue(announceNode.toString(), BtTorrentAnnounce.class);
                    btTorrentAnnounceList.add(announce);
                }
            }
            btTorrentAnnounceList.forEach(btTorrentAnnounce -> {
                        btTorrentAnnounce.setTorrentId(torrentId);
                        btTorrentAnnounceService.insertBtTorrentAnnounce(btTorrentAnnounce);
                    }
            );








        } catch (Exception e) {
            e.printStackTrace();
            // Handle the error (e.g., return null or an error message)

        }
    }

    /**
     * 查询种子主列表
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:list')")
    @PostMapping("/recommend")
    public TableDataInfo recommendTorrents(@RequestParam("user_id") String userId) {
        try {
            // 1. 构造 JSON 请求体
            String jsonRequest = "{\"user_id\": \"" + userId + "\"}";

            // 2. 向 Flask 服务发 POST 请求，返回 JSON 数组
            String jsonResponse = HttpUtils.sendPost(RECOMMEND_API, jsonRequest);

            // 3. 使用 Jackson 解析 JSON 数组为 List<Long>
            ObjectMapper mapper = new ObjectMapper();
            List<Long> idList = mapper.readValue(jsonResponse, new TypeReference<List<Long>>() {});

            // 4. 根据 ID 查询完整种子信息，并保持推荐顺序
            List<BtTorrent> resultList = btTorrentService.selectBtTorrentsByIdsOrdered(idList);

            // 5. 封装成 TableDataInfo 返回
            TableDataInfo rsp = new TableDataInfo();
            rsp.setCode(200);
            rsp.setMsg("推荐成功");
            rsp.setRows(resultList);
            rsp.setTotal(resultList.size());
            return rsp;

        } catch (Exception e) {
            e.printStackTrace();
            TableDataInfo error = new TableDataInfo();
            error.setCode(500);
            error.setMsg("推荐失败：" + e.getMessage());
            return error;
        }
    }

    /**
     * 导出种子主列表
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:export')")
    @Log(title = "种子主", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, BtTorrent btTorrent)
    {
        List<BtTorrent> list = btTorrentService.selectBtTorrentList(btTorrent);
        ExcelUtil<BtTorrent> util = new ExcelUtil<BtTorrent>(BtTorrent.class);
        util.exportExcel(response, list, "种子主数据");
    }

    /**
     * 获取种子主详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:query')")
    @GetMapping(value = "/{torrentId}")
    public AjaxResult getInfo(@PathVariable("torrentId") Long torrentId)
    {
        return success(btTorrentService.selectBtTorrentByTorrentId(torrentId));
    }

    /**
     * 新增种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:add')")
    @Log(title = "种子主", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody BtTorrent btTorrent)
    {
        return toAjax(btTorrentService.insertBtTorrent(btTorrent));
    }

    /**
     * 修改种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:edit')")
    @Log(title = "种子主", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody BtTorrent btTorrent)
    {
        return toAjax(btTorrentService.updateBtTorrent(btTorrent));
    }

    /**
     * 删除种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:remove')")
    @Log(title = "种子主", businessType = BusinessType.DELETE)
	@DeleteMapping("/{torrentIds}")
    public AjaxResult remove(@PathVariable Long[] torrentIds)
    {
        return toAjax(btTorrentService.deleteBtTorrentByTorrentIds(torrentIds));
    }
}
