package com.ruoyi.announce.service.impl;

import com.ruoyi.announce.service.IAnnounceService;
import com.ruoyi.announce.util.BencodeEncoder;
import com.ruoyi.common.core.redis.RedisCache;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Service
public class AnnounceServiceImpl implements IAnnounceService {

    private static final int ANNOUNCE_INTERVAL = 1800; // 秒

    @Autowired
    private RedisCache redisCache;

    @Override
    public Map<String, Object> handleAnnounce(
            byte[] infoHash,
            byte[] peerId,
            int port,
            long uploaded,
            long downloaded,
            long left,
            String event,
            String passkey,
            String ip
    ) throws Exception {
        // 1. 转 hex 作为 Redis key 前缀
        String infoHashHex = bytesToHex(infoHash);
        String peerIdStr = new String(peerId, StandardCharsets.ISO_8859_1);

        // 2. 校验 passkey（可根据业务到 MySQL 查 userId，此处略过）

        // 3. 在 Redis 中记录/刷新此 peer 的信息（TTL 120 秒）
        String peerKey = "peer:" + infoHashHex + ":" + peerIdStr;
        Map<String, Object> peerData = new HashMap<>();
        peerData.put("ip", ip);
        peerData.put("port", port);
        peerData.put("uploaded", uploaded);
        peerData.put("downloaded", downloaded);
        peerData.put("left", left);
        peerData.put("lastSeen", System.currentTimeMillis());
        redisCache.setCacheMap(peerKey, peerData);
        redisCache.expire(peerKey, 1200, TimeUnit.SECONDS);

        // 4. 收集 peers，根据自身状态区分返回哪些 peer
        boolean isSeeder = left == 0;
        Collection<String> keys = redisCache.keys("peer:" + infoHashHex + ":*");
        List<byte[]> peersBin = new ArrayList<>(keys.size());

        for (String key : keys) {
            @SuppressWarnings("unchecked")
            Map<String, Object> data = (Map<String, Object>) redisCache.getCacheMap(key);
            if (data == null) continue;

            String otherPeerId = key.substring(key.lastIndexOf(":") + 1);
            if (otherPeerId.equals(peerIdStr)) continue; // 忽略自己

            String peerIp = (String) data.get("ip");
            int peerPort = ((Number) data.get("port")).intValue();
            long otherLeft = ((Number) data.get("left")).longValue();

            if (peerIp.contains(":")) continue; // 跳过 IPv6

            // 按条件过滤
            if (isSeeder && otherLeft > 0) {
                peersBin.add(encodePeer(peerIp, peerPort));
            } else if (!isSeeder) {
                peersBin.add(encodePeer(peerIp, peerPort));
            }

            if (peersBin.size() >= 50) break;
        }

        // 4.5 合并 peers 到 compact 格式
        ByteArrayOutputStream peerStream = new ByteArrayOutputStream();
        for (byte[] peer : peersBin) {
            if (peer != null && peer.length == 6) {
                peerStream.write(peer);
            }
        }
        byte[] compactPeers = peerStream.toByteArray();

        // 5. 构造返回 Map
        Map<String, Object> reply = new LinkedHashMap<>();
        reply.put("interval", ANNOUNCE_INTERVAL);
        reply.put("min interval", ANNOUNCE_INTERVAL / 2);
        reply.put("complete", countSeeders(infoHashHex));
        reply.put("incomplete", countLeechers(infoHashHex));
        reply.put("peers", compactPeers);

        return reply;
    }


    @Override
    public byte[] encodeBencode(Map<String, Object> reply) throws IOException {
        BencodeEncoder encoder = new BencodeEncoder();
        return encoder.encodeBencode(reply);
    }

    // —— 辅助方法 —— //

    /** 统计 left == 0 的 Seeder 数 */
    private int countSeeders(String infoHashHex) {
        int count = 0;
        for (String key : redisCache.keys("peer:" + infoHashHex + ":*")) {
            @SuppressWarnings("unchecked")
            Map<String, Object> data = (Map<String, Object>) redisCache.getCacheMap(key);
            long left = ((Number) data.get("left")).longValue();
            if (left == 0) count++;
        }
        return count;
    }

    /** 统计 left > 0 的 Leecher 数 */
    private int countLeechers(String infoHashHex) {
        int count = 0;
        for (String key : redisCache.keys("peer:" + infoHashHex + ":*")) {
            @SuppressWarnings("unchecked")
            Map<String, Object> data = (Map<String, Object>) redisCache.getCacheMap(key);
            long left = ((Number) data.get("left")).longValue();
            if (left > 0) count++;
        }
        return count;
    }

    private byte[] encodePeer(String ip, int port) throws Exception {
        if (ip.contains(":")) return null; // 跳过 IPv6

        String[] parts = ip.split("\\.");
        if (parts.length != 4) throw new IllegalArgumentException("无效的 IPv4 地址: " + ip);

        ByteBuffer buf = ByteBuffer.allocate(6);
        for (String part : parts) {
            buf.put((byte) Integer.parseInt(part));
        }
        buf.putShort((short) port);
        return buf.array();
    }



    /** 将字节数组转成十六进制字符串 */
    private static final char[] HEX = "0123456789abcdef".toCharArray();
    private String bytesToHex(byte[] bytes) {
        char[] cs = new char[bytes.length * 2];
        for (int i = 0; i < bytes.length; i++) {
            int v = bytes[i] & 0xFF;
            cs[i * 2]     = HEX[v >>> 4];
            cs[i * 2 + 1] = HEX[v & 0x0F];
        }
        return new String(cs);
    }
}
