package com.ruoyi.tracker.controller;

import java.util.List;

import com.ruoyi.tracker.service.ITrackerProjectUserService;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.tracker.domain.TrackerProjectUser;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 项目与用户关联Controller
 * 
 * @author ruoyi
 * @date 2025-04-20
 */
@RestController
@RequestMapping("/tracker/user")
public class TrackerProjectUserController extends BaseController
{
    @Autowired
    private ITrackerProjectUserService trackerProjectUserService;

    /**
     * 查询项目与用户关联列表
     */
    @PreAuthorize("@ss.hasPermi('system:user:list')")
    @GetMapping("/list")
    public TableDataInfo list(TrackerProjectUser trackerProjectUser)
    {
        startPage();
        List<TrackerProjectUser> list = trackerProjectUserService.selectTrackerProjectUserList(trackerProjectUser);
        return getDataTable(list);
    }

    /**
     * 导出项目与用户关联列表
     */
    @PreAuthorize("@ss.hasPermi('system:user:export')")
    @Log(title = "项目与用户关联", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, TrackerProjectUser trackerProjectUser)
    {
        List<TrackerProjectUser> list = trackerProjectUserService.selectTrackerProjectUserList(trackerProjectUser);
        ExcelUtil<TrackerProjectUser> util = new ExcelUtil<TrackerProjectUser>(TrackerProjectUser.class);
        util.exportExcel(response, list, "项目与用户关联数据");
    }

    /**
     * 获取项目与用户关联详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:user:query')")
    @GetMapping(value = "/{projectId}")
    public AjaxResult getInfo(@PathVariable("projectId") Long projectId)
    {
        return success(trackerProjectUserService.selectTrackerProjectUserByProjectId(projectId));
    }

    /**
     * 新增项目与用户关联
     */
    @PreAuthorize("@ss.hasPermi('system:user:add')")
    @Log(title = "项目与用户关联", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TrackerProjectUser trackerProjectUser)
    {
        return toAjax(trackerProjectUserService.insertTrackerProjectUser(trackerProjectUser));
    }

    /**
     * 修改项目与用户关联
     */
    @PreAuthorize("@ss.hasPermi('system:user:edit')")
    @Log(title = "项目与用户关联", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TrackerProjectUser trackerProjectUser)
    {
        return toAjax(trackerProjectUserService.updateTrackerProjectUser(trackerProjectUser));
    }

    /**
     * 删除项目与用户关联
     */
    @PreAuthorize("@ss.hasPermi('system:user:remove')")
    @Log(title = "项目与用户关联", businessType = BusinessType.DELETE)
	@DeleteMapping("/{projectIds}")
    public AjaxResult remove(@PathVariable Long[] projectIds)
    {
        return toAjax(trackerProjectUserService.deleteTrackerProjectUserByProjectIds(projectIds));
    }
}
