package com.ruoyi.reward.controller;

import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.ArrayList;
import java.util.UUID;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.multipart.MultipartFile;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.reward.domain.RewardTasks;
import com.ruoyi.reward.service.IRewardTasksService;
import com.ruoyi.torrent.domain.BtTorrent;
import com.ruoyi.torrent.domain.BtTorrentAnnounce;
import com.ruoyi.torrent.domain.BtTorrentFile;
import com.ruoyi.torrent.service.IBtTorrentService;
import com.ruoyi.torrent.service.IBtTorrentFileService;
import com.ruoyi.torrent.service.IBtTorrentAnnounceService;
import com.ruoyi.torrent.util.TorrentFileUtil;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * 悬赏任务控制器
 *
 * @author ruoyi
 * @date 2025-05-17
 */
@RestController
@RequestMapping("/reward")
public class RewardTasksController extends BaseController
{
    @Autowired
    private IRewardTasksService rewardTasksService;

    @Autowired
    private IBtTorrentService btTorrentService;

    @Autowired
    private IBtTorrentFileService btTorrentFileService;

    @Autowired
    private IBtTorrentAnnounceService btTorrentAnnounceService;

    private static final String FLASK_URL = "http://49.233.215.144:5000/parse_torrent";
    private static final String TORRENT_PATH = "torrents";
    private static final String BOUNDARY = "----WebKitFormBoundary" + UUID.randomUUID().toString().replace("-", "");

    /**
     * 查询悬赏任务列表
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:list')")
    @GetMapping("/list")
    public TableDataInfo list(RewardTasks rewardTasks)
    {
        startPage();
        List<RewardTasks> list = rewardTasksService.selectRewardTasksList(rewardTasks);
        return getDataTable(list);
    }

    /**
     * 导出悬赏任务列表
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:export')")
    @Log(title = "悬赏任务", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, RewardTasks rewardTasks)
    {
        List<RewardTasks> list = rewardTasksService.selectRewardTasksList(rewardTasks);
        ExcelUtil<RewardTasks> util = new ExcelUtil<RewardTasks>(RewardTasks.class);
        util.exportExcel(response, list, "悬赏任务数据");
    }

    /**
     * 获取悬赏任务详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:query')")
    @GetMapping(value = "/{rewardId}")
    public AjaxResult getInfo(@PathVariable("rewardId") Long rewardId)
    {
        return success(rewardTasksService.selectRewardTasksByRewardId(rewardId));
    }

    /**
     * 新增悬赏任务
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:add')")
    @Log(title = "悬赏任务", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody RewardTasks rewardTasks)
    {
        return toAjax(rewardTasksService.insertRewardTasks(rewardTasks));
    }

    /**
     * 修改悬赏任务
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:edit')")
    @Log(title = "悬赏任务", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody RewardTasks rewardTasks)
    {
        return toAjax(rewardTasksService.updateRewardTasks(rewardTasks));
    }

    /**
     * 删除悬赏任务
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:remove')")
    @Log(title = "悬赏任务", businessType = BusinessType.DELETE)
    @DeleteMapping("/{rewardIds}")
    public AjaxResult remove(@PathVariable Long[] rewardIds)
    {
        return toAjax(rewardTasksService.deleteRewardTasksByRewardIds(rewardIds));
    }

    /**
     * 根据悬赏ID获取Torrent ID
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:query')")
    @GetMapping("/bt/{rewardId}")
    public AjaxResult getRewardBt(@PathVariable("rewardId") Integer rewardId)
    {
        Integer rewardBt = rewardTasksService.selectRewardBtByRewardId(rewardId);
        if (rewardBt == null) {
            return error("未找到对应的 torrent ID");
        }
        return success(rewardBt);
    }

    /**
     * 上传种子文件并关联悬赏任务
     */
    @PreAuthorize("@ss.hasPermi('system:tasks:edit')")
    @Log(title = "悬赏任务种子上传", businessType = BusinessType.INSERT)
    @PostMapping("/uploadTorrent")
    @Transactional
    public AjaxResult uploadTorrent(@RequestParam("file") MultipartFile file, @RequestParam("reward_id") Integer rewardId)
    {
        try {
            // 验证文件类型
            if (!file.getOriginalFilename().endsWith(".torrent")) {
                return AjaxResult.error("仅支持 .torrent 文件");
            }

            // 验证 reward_id
            RewardTasks task = rewardTasksService.selectRewardTasksByRewardId(rewardId.longValue());
            if (task == null) {
                return AjaxResult.error("无效的悬赏任务ID");
            }
            if (!"2".equals(task.getStatus())) {
                return AjaxResult.error("悬赏任务未完成，无法上传种子文件");
            }
            if (!task.getAccepterId().equals(SecurityUtils.getUserId())) {
                return AjaxResult.error("只有任务接受者可以上传种子文件");
            }

            // 检查是否已关联 torrent
            Integer existingBt = rewardTasksService.selectRewardBtByRewardId(rewardId);
            if (existingBt != null) {
                return AjaxResult.error("该悬赏任务已关联种子文件");
            }

            // 调用 Flask 服务解析种子文件
            HttpURLConnection connection = (HttpURLConnection) new URL(FLASK_URL).openConnection();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setConnectTimeout(5000);
            connection.setReadTimeout(5000);
            connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + BOUNDARY);

            // 写入文件数据
            try (OutputStream outputStream = connection.getOutputStream()) {
                outputStream.write(("--" + BOUNDARY + "\r\n").getBytes(StandardCharsets.UTF_8));
                outputStream.write(("Content-Disposition: form-data; name=\"torrent\"; filename=\"" + file.getOriginalFilename() + "\"\r\n").getBytes(StandardCharsets.UTF_8));
                outputStream.write(("Content-Type: application/octet-stream\r\n\r\n").getBytes(StandardCharsets.UTF_8));
                file.getInputStream().transferTo(outputStream);
                outputStream.write("\r\n".getBytes(StandardCharsets.UTF_8));
                outputStream.write(("--" + BOUNDARY + "--\r\n").getBytes(StandardCharsets.UTF_8));

                // 处理 Flask 响应
                int responseCode = connection.getResponseCode();
                if (responseCode != 200) {
                    return AjaxResult.error("与Flask服务通信失败，响应码: " + responseCode);
                }

                // 保存文件到服务器
                String filePath = TORRENT_PATH + "/" + SecurityUtils.getUserId() + "/" + file.getOriginalFilename();
                if (filePath.contains("..")) {
                    return AjaxResult.error("无效的文件路径");
                }
                TorrentFileUtil.uploadFile(file, filePath);

                // 解析响应并存储到数据库
                String responseBody = new String(connection.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
                Long torrentId = parseTorrentData(responseBody, filePath);

                // 插入 reward_bt 记录
                rewardTasksService.insertRewardBt(rewardId, torrentId.intValue());

                return AjaxResult.success("种子文件上传并处理成功，torrent_id: " + torrentId);
            }
        } catch (IOException e) {
            logger.error("文件上传或与Flask服务通信失败: {}", e.getMessage());
            return AjaxResult.error("文件上传或与Flask服务通信失败: " + e.getMessage());
        }
    }

    /**
     * 解析种子数据并存储到数据库
     */
    private Long parseTorrentData(String responseData, String filePath)
    {
        try {
            ObjectMapper objectMapper = new ObjectMapper();
            JsonNode jsonNode = objectMapper.readTree(responseData);
            JsonNode btTorrentNode = jsonNode.get("btTorrent");
            JsonNode btTorrentAnnounceNode = jsonNode.get("btTorrentAnnounce");
            JsonNode btTorrentFilesNode = jsonNode.get("btTorrentFiles");

            // 存储 bt_torrent
            BtTorrent btTorrent = objectMapper.readValue(btTorrentNode.toString(), BtTorrent.class);
            btTorrent.setCreatedBy(SecurityUtils.getUsername());
            btTorrent.setUploaderId(SecurityUtils.getUserId());
            btTorrent.setFilePath(filePath);
            btTorrentService.insertBtTorrent(btTorrent);
            Long torrentId = btTorrent.getTorrentId();

            // 存储 bt_torrent_file
            List<BtTorrentFile> btTorrentFiles = objectMapper.readValue(
                    btTorrentFilesNode.toString(),
                    new TypeReference<List<BtTorrentFile>>() {});
            btTorrentFiles.forEach(btFile -> {
                btFile.setTorrentId(torrentId);
                btTorrentFileService.insertBtTorrentFile(btFile);
            });

            // 存储 bt_torrent_announce
            List<BtTorrentAnnounce> btTorrentAnnounceList = new ArrayList<>();
            if (btTorrentAnnounceNode.isArray()) {
                for (JsonNode announceNode : btTorrentAnnounceNode) {
                    BtTorrentAnnounce announce = objectMapper.readValue(announceNode.toString(), BtTorrentAnnounce.class);
                    btTorrentAnnounceList.add(announce);
                }
            }
            btTorrentAnnounceList.forEach(btTorrentAnnounce -> {
                btTorrentAnnounce.setTorrentId(torrentId);
                btTorrentAnnounceService.insertBtTorrentAnnounce(btTorrentAnnounce);
            });

            return torrentId;
        } catch (Exception e) {
            logger.error("解析种子数据失败: {}", e.getMessage());
            throw new RuntimeException("解析种子数据失败: " + e.getMessage());
        }
    }
}